// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


// The modeule converts a 32-bit data bus AVMM I/F into a native 
// GDR 8-bit data bus AVMM I/F. The msb of output addr is the indication
// of DWord_Access
 
module ft_avmm_32to8_bridge 
#(  
    parameter READ_PIPELINE_ENABLE = 0, 
    parameter ADDR_WIDTH = 18
) (
   // AVMM slave Port
   input  logic                  i_clk                 ,// 
   input  logic                  i_rst                 ,// 
   
   input  logic [ADDR_WIDTH-1:0] i_avmm_s32_addr           ,// 
   input  logic [31:0]           i_avmm_s32_wdata          ,//
   input  logic                  i_avmm_s32_write          ,//
   input  logic                  i_avmm_s32_read           ,//
   input  logic [3:0]            i_avmm_s32_byte_enable    ,//
   output logic [31:0]           o_avmm_s32_readdata       ,//
   output logic                  o_avmm_s32_waitrequest    ,//
   output logic                  o_avmm_s32_readdatavalid  ,// 
 
   // AVMMM master Port
   output  logic [ADDR_WIDTH+2:0] o_avmm_m8_addr           ,//  <1-bit DW_ACC>, <s32_addr>, <2-bit byte addr>
   output  logic [7:0]            o_avmm_m8_wdata          ,//
   output  logic                  o_avmm_m8_write          ,//
   output  logic                  o_avmm_m8_read           ,//
   input   logic [7:0]            i_avmm_m8_readdata       ,//
   input   logic                  i_avmm_m8_waitrequest     //
);
 
logic [1:0] byte_end, cur_byte, next_byte;  // to represent byteenables
logic       dw_acc;              // 32-bit DWord_access
logic [31:0] dw_data, next_dw_data ;
 
logic [2:0] cur_st, next_st;       // idle, read, write, read_done, or write_done, 
 
localparam ST_IDLE        = 3'b000;
localparam ST_READ        = 3'b001;
localparam ST_READ_PAUSE  = 3'b010;
localparam ST_READ_DONE   = 3'b011;
localparam ST_WRITE       = 3'b100;
localparam ST_WRITE_PAUSE = 3'b101;
localparam ST_WRITE_DONE  = 3'b110;
 
 
assign o_avmm_m8_addr  = {dw_acc, i_avmm_s32_addr[ADDR_WIDTH-1:0], cur_byte};
assign o_avmm_m8_wdata = (cur_st==ST_WRITE)? dw_data[cur_byte*8+:8] : 8'h0;
assign o_avmm_m8_write = (cur_st==ST_WRITE)? 1'b1 : 1'b0;
assign o_avmm_m8_read  = (cur_st==ST_READ)? 1'b1 : 1'b0;
 
assign o_avmm_s32_readdata    = (cur_st==ST_READ_DONE || (READ_PIPELINE_ENABLE && o_avmm_s32_readdatavalid))?
                                dw_data[31:0] : 32'h0;
assign o_avmm_s32_waitrequest = (cur_st==ST_READ_DONE || cur_st==ST_WRITE_DONE)? 1'b0 : 1'b1;
 
 
// convertion FSM 
//
 
always_ff @(posedge i_clk or posedge i_rst) begin
    if (i_rst) begin
        cur_st     <= ST_IDLE;
	dw_data    <= 32'h0;
       	cur_byte   <= 2'b00;   
        o_avmm_s32_readdatavalid <= 1'b0;
    
	dw_acc     <= 1'b0;
	byte_end   <= 2'b00;
    end else begin   // non-reset
	cur_st     <= next_st;
       	cur_byte   <= next_byte;
        dw_data    <= next_dw_data;
	o_avmm_s32_readdatavalid  <= ( READ_PIPELINE_ENABLE && (cur_st==ST_READ_DONE) )? 1'b1 : 1'b0;
	
	if (cur_st==ST_IDLE) begin
	    if (i_avmm_s32_write || i_avmm_s32_read)  begin
		dw_acc <= ( (i_avmm_s32_byte_enable==4'hf) || (i_avmm_s32_byte_enable==4'h0) )? 1'b1 : 1'b0;
		byte_end <= (i_avmm_s32_byte_enable[3])? 2'b11 : (i_avmm_s32_byte_enable[2])? 2'b10 : 
			(i_avmm_s32_byte_enable[1])? 2'b01 : (i_avmm_s32_byte_enable[0])? 2'b00 : 2'b11;
	     end  else begin	// no access
	             dw_acc       <= 1'b0;
		     byte_end     <= 2'b00;
             end 
	end       // ST_IDLE
    end           // non-reset
end     // always
 
 
always_comb  begin
    next_st      = cur_st;
    next_byte    = cur_byte;
    next_dw_data = dw_data;
 
    case (cur_st) 
	ST_IDLE: begin
		if (i_avmm_s32_write || i_avmm_s32_read) begin
			next_st = (i_avmm_s32_write)? ST_WRITE : ST_READ;
		        next_byte = (i_avmm_s32_byte_enable[0])? 2'b00 : (i_avmm_s32_byte_enable[1])? 2'b01 : 
			            (i_avmm_s32_byte_enable[2])? 2'b10 : (i_avmm_s32_byte_enable[3])? 2'b11 : 2'b00;
		        next_dw_data = (i_avmm_s32_write)? i_avmm_s32_wdata : 32'h0;
		    end   // read or write
	         end
 
	ST_READ: begin
		if ( ~i_avmm_m8_waitrequest ) begin
			next_st = (cur_byte==byte_end)? ST_READ_DONE : ST_READ_PAUSE;
			next_dw_data[cur_byte*8+:8] = i_avmm_m8_readdata;
		    end
	         end
 
	 ST_READ_PAUSE: begin
			next_st = ST_READ;
			next_byte = cur_byte + 1'b1;
	         end
 
	 ST_READ_DONE: begin
			next_st = ST_IDLE;
	         end
 
	 ST_WRITE: begin
		if ( ~i_avmm_m8_waitrequest ) begin
			next_st = (cur_byte==byte_end)? ST_WRITE_DONE : ST_WRITE_PAUSE;
		    end
	         end
 
	 ST_WRITE_PAUSE: begin
			next_st = ST_WRITE;
			next_byte = cur_byte + 1'b1;
	         end
 
	 ST_WRITE_DONE: begin
			next_st = ST_IDLE;
	         end
 
    endcase  
 
end    // always_comb
 
 
 
endmodule: ft_avmm_32to8_bridge
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "ZCs+52HNmv7JxZwiTjf2MarrP/6s5+GMMeVNBgHmlFk8kA5+3C1CMZ/Rdl0jFreT2FsLH0CzN799DuR3pC98S/dCKaEszy50+yM8RYMlo/2rGIeZTlMiI6Sy1fetFN7LknfaKoGjtWLqDAU18+sdF8KEoSMFTy6tHcGfuhmJtWBHaG9RxbVSrYkZBTOymRlKVEHea3vzMvSPgOVRc6UrqsiiWHg/8AoPp3s5ICPhdfB40n2xjn7eYvJytQAp1nbioPcDNRE0RsJoAAHIJQ7LG7VXr2i3aL/2ORlJv5t22nDGhnhfjULnhSXTgUXUtQ1Qw6dd0Y81wCBQer7PB+ilZZLTZqTaeHPNkvD7JcGGn5hrzKfVGVulP5FjgtldOYjxc7z4tJ3zkBm7BsSZLB7VAFg8nlyqj1LvMjzjNUpUoX1yQjxoNwDQTq79crtrCPbCSATKvDcqP0yQQlAvW/GduUATpxpsAsJLH30aHBtnIFgGTcGfs/OfXc8tgZNJuiMDiXQvwksV92IyjzOkzoyIj49+uZau/tc7gNnW+zmm4jDoT9RLftMbjOejBDNlHtj2CDIdeIyMM4D8Av+WVN9Ng923TRB1l6bY75fzO0jP8T1FF2Ltym4CXDhrxgEL8yCPLwK1EeEdUIYPun8V5GN9OLbocrEgKR9h1IDu9yl+9huPe+XDVGOFEWooZSDCVp7J2/A8jmj84fYWCJdceu0c3rwvKttIIAXd32bcLAv9U5wqripx1L0zT8AoTU/ytKn7yt4+WOgwmB7+A9fm/4r5gAOgk6GA/bdTaqjdX4qvWYuhbG77ggP9szkQ/V76sJUP2ZPV//PzvwZzGzZDLlrairAtn1O10ucc7EVRpiHiYFZvUliX7YoHSJcMd2dwZRgfTpgZFjqSqZgQaskYr9t+bhQPfL9FK8bp/e6qmC3G4XVQdXEZrENVVWII+T/msf6ONblt+wMC6eS07sNCTLOYcN6xqe1cwhtk//QXdZsAeQ3xnV3RXIReGjmnX16ihZmn"
`endif