// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.



//----------------------------------------------------------------------------- 
//  Project Name:  GDR Dynamic Reconfiguration (F-tile)                                  
//  Module Name :  dr_f_avmm_arbiter_2to1                                  
//  Author ID   :  hafeezmu                                   
//  Date        :  Wed Jan 29 11:07:44 2020   
//  Version     :  v1.0 (Initial Draft)                               
//  Description :  GDR DR AVMM Arbiter 2to1 for F-tile. This component is part 
//                 of DR SIP. This DR AVMM Arbiter is used between two Host and 
//                 one AVMM slave interface                                             
//-----------------------------------------------------------------------------
// synthesis translate_off  
`timescale 1 ps / 1 fs
// synthesis translate_on
module dr_f_avmm_arbiter_2to1 #(
    parameter TIMEOUT       = 1000,
    parameter DWIDTH        = 32,
    parameter AWIDTH        = 32
) (
    input   logic                   csr_clk,
    input   logic                   rst_n,
    input   logic                   timeout_en,
    input   logic   [31:0]          avmm_timeout,
    
    // Master 0
    input   logic                   m0_read,
    input   logic                   m0_write,
    input   logic   [AWIDTH-1:0]    m0_address,
    input   logic   [DWIDTH-1:0]    m0_write_data,
    output  logic   [DWIDTH-1:0]    m0_read_data,
    output  logic                   m0_read_valid,
    output  logic                   m0_wait_request,
    output  logic                   m0_timeout,

    // Master 1
    input   logic                   m1_read,
    input   logic                   m1_write,
    input   logic   [AWIDTH-1:0]    m1_address,
    input   logic   [DWIDTH-1:0]    m1_write_data,
    output  logic   [DWIDTH-1:0]    m1_read_data,
    output  logic                   m1_read_valid,
    output  logic                   m1_wait_request,
    output  logic                   m1_timeout,

    // Slave
    output  logic                   s_read,
    output  logic                   s_write,
    output  logic   [AWIDTH-1:0]    s_address,
    output  logic   [DWIDTH-1:0]    s_write_data,
    input   logic   [DWIDTH-1:0]    s_read_data,
    input   logic                   s_read_valid,
    input   logic                   s_wait_request
);

//-----------------------------------------------------------------------------
//Declarations
//-----------------------------------------------------------------------------

    localparam TIMER_MAX = TIMEOUT + 10;    // Extra cycles for skid room
    localparam TIMER_BITS = $clog2(TIMER_MAX + 1);

    typedef enum {
        IDLE,
        SLAVE_WAIT,
        SLAVE_WAIT_VALID,
        SLAVE_DONE,
        SLAVE_READ_RETURN
    } STATE_e;

    typedef enum {
        TYPE_NONE,
        TYPE_READ,
        TYPE_WRITE
    } TRANS_TYPE_e;

    STATE_e         state, state_next;
    logic           active_master, active_master_next;
    TRANS_TYPE_e    transaction_type, transaction_type_next;

    logic   [31:0]  timer, timer_next;
    logic           timed_out;

//-----------------------------------------------------------------------------
//Main Body of Code
//-----------------------------------------------------------------------------

    always @ (posedge csr_clk) begin
        if (!rst_n) begin
            state               <= IDLE;
            active_master       <= 1'b0;
            transaction_type    <= TYPE_NONE;
            timer               <= {32{1'b0}};
        end else begin
            state               <= state_next;
            active_master       <= active_master_next;
            transaction_type    <= transaction_type_next;
            timer               <= timer_next;
        end
    end

    always @ (*) begin
        if (state == IDLE) begin
            timer_next = '0;
        end else begin
            if (timeout_en & (|avmm_timeout)) begin
                timer_next = timer + 1'd1;
            end else begin
                timer_next = '0;
            end
        end
    end

    assign timed_out = (timer > avmm_timeout);

    always @ (*) begin: AVMM_ARIBTER_SM
        state_next              = state;
        active_master_next      = active_master;
        transaction_type_next   = transaction_type;
        case (state)
            IDLE : begin
                if (m0_read || m1_read) begin
                    state_next = SLAVE_WAIT;
                    active_master_next = m0_read ? 1'd0 : 1'd1;
                    transaction_type_next   = TYPE_READ;
                end
                if (m0_write || m1_write) begin
                    state_next = SLAVE_WAIT;
                    active_master_next = m0_write ? 1'd0 : 1'd1;
                    transaction_type_next   = TYPE_WRITE;
                end
            end
            SLAVE_WAIT : begin
                if (!s_wait_request || timed_out) begin
                    case (transaction_type)
                        TYPE_READ  : state_next = SLAVE_READ_RETURN;
                        TYPE_WRITE : state_next = SLAVE_DONE;
                        default    : state_next = SLAVE_DONE;
                    endcase
                end
            end
            SLAVE_WAIT_VALID : if (s_read_valid || timed_out) state_next = SLAVE_READ_RETURN;
            SLAVE_DONE       : begin
                state_next = IDLE;
                transaction_type_next   = TYPE_NONE;
            end
            SLAVE_READ_RETURN : begin
                state_next = IDLE;
                transaction_type_next   = TYPE_NONE;
            end
            default : begin
                state_next              = IDLE;
                transaction_type_next   = TYPE_NONE;
                active_master_next      = 1'd0;
            end
        endcase
    end: AVMM_ARIBTER_SM

    always @ (posedge csr_clk) begin
        if (s_read_valid) begin
            m0_read_data <= s_read_data;
            m1_read_data <= s_read_data;
        end
    end

    always @ (posedge csr_clk) begin
        if (!rst_n) begin
            m0_read_valid <= 1'b0;
            m1_read_valid <= 1'b0;
        end
        else begin
            m0_read_valid <= (s_read_valid || (timed_out && (transaction_type == TYPE_READ))) && (active_master == 1'd0);
            m1_read_valid <= (s_read_valid || (timed_out && (transaction_type == TYPE_READ))) && (active_master == 1'd1);
        end
    end     
    //assign m0_read_valid = (state == SLAVE_READ_RETURN) && (active_master == 1'd0);
    //assign m1_read_valid = (state == SLAVE_READ_RETURN) && (active_master == 1'd1);

    assign m0_wait_request = !((state == SLAVE_WAIT) && (active_master == 1'b0) && (!s_wait_request || timed_out));
    assign m1_wait_request = !((state == SLAVE_WAIT) && (active_master == 1'b1) && (!s_wait_request || timed_out));

    assign s_read  = (state == SLAVE_WAIT) && (transaction_type == TYPE_READ);
    assign s_write = (state == SLAVE_WAIT) && (transaction_type == TYPE_WRITE);

    assign s_address    = (active_master == 1'd0) ? m0_address    : m1_address;
    assign s_write_data = (active_master == 1'd0) ? m0_write_data : m1_write_data;

    assign m0_timeout   = timed_out && (active_master == 1'b0);
    assign m1_timeout   = timed_out && (active_master == 1'b1);

endmodule //DR_F_AVMM_ARBITER_2to1

`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "79ztKTnV2ny/eHpgZYepe9G9wwwdeg9hTOdIyLdh4JTuZZ1DT50cJOPNpZ0w+c5aVHkra3Tx9+PMlDB4omAPannbtZoJG9ZekMCcgtPHzQ4GWiJfmhYoHZqap4yb9/uKlHSl9TpAcV1FKEwkrctIKziGBKSQ0S70p/jW9wD4CtM3X7P+ggV0x7Prg108lDSsdB1kGvgcQN/5kLCIPO6i12claBXdWuhbdU7Bm3JHKTrEHvsWi6NDZgBvceTKVVIdnCWQQL8a53EqfSn4KV4EPaciFA09WLV6JpFAxlQvYHruBQWQp1KTz06McYmGuWbrjfNLNt2Po26jfSxZv3R7JGkBoGC3t3sA3Au770c9OHp3XVM0gYhPOUZ7wIayVdeu33ecAhA/l4qe+RVEmcX2f3BMtd4yTDO5ndU5tzObpvOVGZ2tj+RHZdYWtet9c3sFO4Cj6YbTCeTZz9i0s/3N+lRuWbiXWcGu6iettMK5OZ8jvFJITTPumA4Oojh3iAQ/dn9y1QDwntMG01r1ijKd4ze1IOyTD8l/kllx6U32FJuVUCjEQdxKEHIjWePlj9+I/JEH2shPN6PF43XBWpA9GhunFezw9A1avq3Q83OFK1CvAnoe7+u69MSGYfGBjEB+Iwkci+qGuvCgu8MzZqL2vYi2xGrlLLfOmVgB+y3QkKkIdnwlotwVX2EuniDqtcGGe/47ABu6Q3N1wWABhZfsaApgQx9SelW0sZB+wDa0JIC7s2vXCuLs9cQrJZaQls7fllh5oNoCFzr5UqVfApv3wrX99+bZ6wKP8aZ0nXKu3uXYRGkN86AOWLRuz2d5kTh4qucLn15kUpDbMdRvJriHiEG+3Hfffz16iCfG0kEPVEF5IezhopQm5MGofh5JEjgiCU2D/t2UY8GVdBSqPGlVKuYKTfg5QBr8JXap4KLDNQP9IRxjx06w3d1txtdg8JuLQ2QwgsIcDnqSQt3qzgUvKy7Qj6zkhd/BJu8HbW61V9G7KLhP4wVc7/o69Tko+SD4"
`endif