// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


//----------------------------------------------------------------------------- 
//  Project Name:  GDR Dynamic Reconfiguration (F-tile)                                  
//  Module Name :  dr_f_clock_crosser                                  
//  Author ID   :  hafeezmu                                   
//  Date        :  Thu July 30 16:00:46 2020   
//  Version     :  v1.0 (Initial Draft)                               
//  Description :  GDR DR AVMM to memory clock crosser for data is guaranteed    
//                 valid by control signal clock crossing. Cut data buffer 
//                 false path.                                             
//-----------------------------------------------------------------------------  
// synthesis translate_off
`timescale 1 ps / 1 fs
// synthesis translate_on
module dr_f_clock_crosser #(
     parameter  DATA_WIDTH          = 32,
     parameter  FORWARD_SYNC_DEPTH  = 2,
     parameter  BACKWARD_SYNC_DEPTH = 2
 ) (
     input   logic                   in_clk,
     input   logic                   in_reset,
     output  logic                   in_ready,
     input   logic                   in_valid,
     input   logic [DATA_WIDTH-1:0]  in_data,

     input   logic                   out_clk,
     input   logic                   out_reset,
     input   logic                   out_ready,
     output  logic                   out_valid,
     output  logic [DATA_WIDTH-1:0]  out_data
 );

  // Data is guaranteed valid by control signal clock crossing.  Cut data
  // buffer false path.
  (* altera_attribute = {"-name SUPPRESS_DA_RULE_INTERNAL \"D101,D102\""} *) reg [DATA_WIDTH-1:0] in_data_buffer;
  reg    [DATA_WIDTH-1:0] out_data_buffer;

  reg                     in_data_toggle;
  wire                    in_data_toggle_returned;
  wire                    out_data_toggle;
  reg                     out_data_toggle_flopped;

  wire                    take_in_data;
  wire                    out_data_taken;

  wire                    out_valid_internal;
  wire                    out_ready_internal;

  assign in_ready = ~(in_data_toggle_returned ^ in_data_toggle);
  assign take_in_data = in_valid & in_ready;
  assign out_valid_internal = out_data_toggle ^ out_data_toggle_flopped;
  assign out_data_taken = out_ready_internal & out_valid_internal;

  always @(posedge in_clk or posedge in_reset) begin
    if (in_reset) begin
      in_data_buffer <= 'b0;
      in_data_toggle <= 1'b0;
    end else begin
      if (take_in_data) begin
        in_data_toggle <= ~in_data_toggle;
        in_data_buffer <= in_data;
      end
    end //in_reset
  end //in_clk always block

  always @(posedge out_clk or posedge out_reset) begin
    if (out_reset) begin
      out_data_toggle_flopped <= 1'b0;
      out_data_buffer <= 'b0;
    end else begin
      out_data_buffer <= in_data_buffer;
      if (out_data_taken) begin
        out_data_toggle_flopped <= out_data_toggle;
      end
    end //end if
  end //out_clk always block
  
    dr_f_xcvr_resync_std #(
        .SYNC_CHAIN_LENGTH  (FORWARD_SYNC_DEPTH),
        .WIDTH              (1),
        .INIT_VALUE         (0)
    ) in_to_out_synchronizer (
        .clk                (out_clk),
        .reset              (out_reset),
        .d                  (in_data_toggle),
        .q                  (out_data_toggle)
    );
    
    dr_f_xcvr_resync_std #(
        .SYNC_CHAIN_LENGTH  (BACKWARD_SYNC_DEPTH),
        .WIDTH              (1),
        .INIT_VALUE         (0)
    ) out_to_in_synchronizer (
        .clk                (in_clk),
        .reset              (in_reset),
        .d                  (out_data_toggle_flopped),
        .q                  (in_data_toggle_returned)
    );

    assign out_valid = out_valid_internal;
    assign out_ready_internal = out_ready;
    assign out_data = out_data_buffer;

endmodule
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "79ztKTnV2ny/eHpgZYepe9G9wwwdeg9hTOdIyLdh4JTuZZ1DT50cJOPNpZ0w+c5aVHkra3Tx9+PMlDB4omAPannbtZoJG9ZekMCcgtPHzQ4GWiJfmhYoHZqap4yb9/uKlHSl9TpAcV1FKEwkrctIKziGBKSQ0S70p/jW9wD4CtM3X7P+ggV0x7Prg108lDSsdB1kGvgcQN/5kLCIPO6i12claBXdWuhbdU7Bm3JHKTrnkDRkPj0OhtFFHpA881NyQQMbqwDteYX5Lz5CRrAp1Mm8lTMIsfg9xwLkE2KzfDCkm3N4mCsqbuUvCj89xUvJW0CT18+G3E9s2C3knzEE0bm+smxHwKcHAgFtLNj78o0Djz6Fdlk52gc1DHb3dWNYZWb123uAp0rZftSi9SOQGKe3BXhVmcP1QA2yGSgere3iIwS1q+EqpEEq5wsv8JgfQZJWTsFJSbVSxP1JojAP85HxO//em4c6Se55YJDZDOJucaUrmAL7RaXy1YeQ7wvUG5OgmfWN1fIcxFxPRrvqrikp2e8OSSC2sUkPOzOupvXPo93b7mVN6VfaAytxcPjNfYbIkTMOJLhVFFP8/gnyhOIlmh6PziFlHbkWBLazMa57B7kmvwBq840QcldFW156R7O02jjBV4H5xAu0ncKkZPlRjXHP7X40tbSZ8mtxhNdSU/VhsBOzMLIv+o7HAUTHgVV8W8NJG8+rieYkVis5SxoxVququ3tzmFMb5ObKBam6j+TM9wtrGOzceP+9gMsItjKAKwzC/LMKrVUlY5I5Xe9ygkMuMMo8tCXtmNsgxi5DLjBUn4J/X/Xuqp0FsXJOgNptrmE8ZzIdzjvTFhE1myMnGCwfFD64hNmXr/dFYvUN2hNku0HoW7yAdEyiifLiSx2c/I0D/L3RKB5ocHC9WLk8o57xL+KySw77PqOn+5qe068j35rfW+lLfidyhWiJjHdu9TGNQ5j3iF7R5MecPLrUCKoQmFmIi0pGHVAnV65cVtQ+bAiXj0qAyJvPlOER"
`endif