// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


//----------------------------------------------------------------------------- 
//  Project Name:  GDR Dynamic Reconfiguration (F-tile)                                  
//  Module Name :  dr_f_gavmm_32to8                                  
//  Author ID   :  hafeezmu                                   
//  Date        :  Wed May 06 10:17:34 2020   
//  Version     :  v1.0 (Initial Draft)                               
//  Description :  GDR DR 32bits to 8bits AVMM data conversion block for F-tile. 
//                 This component is top block of DR SIP that covert the Global
//                 AVMM accesses from NIOS 32bits to Hard IP 8bits                                            
//-----------------------------------------------------------------------------
// synthesis translate_off
`timescale 1 ps / 1 fs
// synthesis translate_on

module dr_f_gavmm_32to8 #(
    parameter TIMEOUT = 1000
) (
    input               clk,
    input               rst_n,
    input       [31:0]  avmm_timeout,    
    output reg          avmm_error,
    
    input               master_read,
    input               master_write,
    input       [31:0]  master_address,
    input       [31:0]  master_write_data,
    input       [3:0]   master_byte_enable,
    output reg          master_read_valid,
    output reg          master_wait,
    output reg  [31:0]  master_read_data,

    input               slave_enable,
    output reg          slave_read,
    output reg          slave_write,
    output reg  [31:0]  slave_address,
    output reg  [ 7:0]  slave_write_data,
    input               slave_read_valid,
    input               slave_wait,
    input       [ 7:0]  slave_read_data
);

    localparam TIMER_BITS = $clog2(TIMEOUT + 1);
    localparam PAGE_ADDRESS = 32'hF_FFFC;
    
    localparam IDLE             = 5'd0;
    localparam WR_WAIT_P        = 5'd1;
    localparam WR_WAIT_0        = 5'd2;
    localparam WR_WAIT_1        = 5'd3;
    localparam WR_WAIT_2        = 5'd4;
    localparam WR_WAIT_3        = 5'd5;
    localparam WR_DONE          = 5'd6;
    localparam RD_WAIT_ACCEPT_P = 5'd7;
    localparam RD_WAIT_ACCEPT_0 = 5'd8;
    localparam RD_WAIT_ACCEPT_1 = 5'd9;
    localparam RD_WAIT_ACCEPT_2 = 5'd10;
    localparam RD_WAIT_ACCEPT_3 = 5'd11;
    localparam RD_WAIT_VALID_0  = 5'd12;
    localparam RD_WAIT_VALID_1  = 5'd13;
    localparam RD_WAIT_VALID_2  = 5'd14;
    localparam RD_WAIT_VALID_3  = 5'd15;
    localparam RD_ACCEPT        = 5'd16;
    localparam RD_VALID         = 5'd17;
    reg [4:0]               state;    

    reg [31:0]              timer;
    reg                     slave_enable_r;
    wire                    slave_enable_redge;
    reg                     master_write_r;
    reg                     master_read_r;

    reg [4:0]               prev_page_addr;
    reg                     prev_page_addr_valid;

`ifndef ALTERA_RESERVED_QIS
    // Simulation hook for enabling DR_F_GAVMM_SPEEDUP
    reg                     master_read_reg;
    wire                    master_read_write_combined;

    always_ff @(posedge clk or negedge rst_n) begin
        if (~rst_n) begin
            master_read_reg <= 1'b0;
        end
        else begin
            if (master_read) begin
                master_read_reg <= 1'b1;
            end else if (master_read_valid) begin
                master_read_reg <= 1'b0;
            end else begin
                master_read_reg <= master_read_reg;
            end
        end
    end
    
    assign master_read_write_combined = master_read | master_read_reg | master_write;
`endif // ifndef ALTERA_RESERVED_QIS

    always_ff @(posedge clk) begin
        if (~rst_n) begin
            slave_enable_r   <= 1'b0;
        end else begin
            slave_enable_r   <= slave_enable;
        end
    end

    always_ff @(posedge clk) begin
        if (~rst_n) begin
            avmm_error <= 1'b0;
        end 
        else begin
            if ((timer == avmm_timeout) & (|avmm_timeout)) begin
                avmm_error <= 1'b1;
            end
            else begin
                avmm_error <= 1'b0;
            end
        end
    end
    
    assign slave_enable_redge = slave_enable & ~slave_enable_r;

    always_ff @(posedge clk) begin
        if (~rst_n) begin
            state               <= IDLE;
            master_read_valid   <= 1'b0;
            master_wait         <= 1'b1;
            master_read_data    <= 32'd0;
            slave_read          <= 1'b0;
            slave_write         <= 1'b0;
            slave_address       <= 32'd0;
            slave_write_data    <= 8'd0;
            timer               <= {32{1'b0}};
            master_write_r      <= 1'b0;
            master_read_r       <= 1'b0;
            prev_page_addr_valid<= 1'b0;
            prev_page_addr      <= 5'b0;
        end else begin
            state               <= state;
            master_read_valid   <= master_read_valid;
            master_wait         <= master_wait;
            master_read_data    <= master_read_data;
            slave_read          <= slave_read;
            slave_write         <= slave_write;
            slave_address       <= slave_address;
            slave_write_data    <= slave_write_data;
            master_write_r      <= master_write_r;
            master_read_r       <= master_read_r;

            // Timeout handler
            if ((state != IDLE) && ~slave_enable_redge) begin
                timer   <= timer + 1'b1;
                if ((timer == avmm_timeout) & (|avmm_timeout)) begin
                    master_wait         <= 1'b0;
                    slave_write         <= 1'b0;
                    slave_read          <= 1'b0;
                    if (master_write_r) begin
                        state               <= WR_DONE;
                    end else if (master_read_r) begin
                        state               <= RD_ACCEPT;
                    end
                end
            end else begin
                timer   <= {TIMER_BITS{1'b0}};
            end

            case (state)
                IDLE                : begin
                    if (master_write) begin
                        master_read_valid        <= 1'b0;
                        master_wait              <= 1'b1;
                        master_read_data         <= 32'd0;
                        slave_read               <= 1'b0;
                        master_write_r           <= 1'b1;
                        master_read_r            <= 1'b0;
                        if (|master_byte_enable & (~prev_page_addr_valid | (prev_page_addr_valid & (master_address[24:20] != prev_page_addr)))) begin
                            state                       <= WR_WAIT_P;
                            slave_address               <= PAGE_ADDRESS;
                            slave_write_data            <= {3'b000,master_address[24:20]}; //Page Value
                            slave_write                 <= slave_enable ? 1'b1 : 1'b0;
                            prev_page_addr              <= master_address[24:20];
                            prev_page_addr_valid        <= 1'b1;
                        end else begin
                            casez (master_byte_enable)
                                4'b???1 : begin
                                    state               <= WR_WAIT_0;
                                    slave_address       <= {12'h1,master_address[19:2], 2'd0};
                                    slave_write_data    <= master_write_data[7:0];
                                    slave_write         <= slave_enable ? 1'b1 : 1'b0;
                                end
                                4'b??10 : begin
                                    state               <= WR_WAIT_1;
                                    slave_address       <= {12'h1,master_address[19:2], 2'd1};
                                    slave_write_data    <= master_write_data[15:8];
                                    slave_write         <= slave_enable ? 1'b1 : 1'b0;
                                end
                                4'b?100 : begin
                                    state               <= WR_WAIT_2;
                                    slave_address       <= {12'h1,master_address[19:2], 2'd2};
                                    slave_write_data    <= master_write_data[23:16];
                                    slave_write         <= slave_enable ? 1'b1 : 1'b0;
                                end
                                4'b1000 : begin
                                    state               <= WR_WAIT_3;
                                    slave_address       <= {12'h1,master_address[19:2], 2'd3};
                                    slave_write_data    <= master_write_data[31:24];
                                    slave_write         <= slave_enable ? 1'b1 : 1'b0;
                                end
                                4'b0000 : begin
                                    state               <= WR_DONE;
                                    master_wait         <= 1'b0;
                                    slave_write         <= 1'b0;
                                end
                            endcase
                        end
                    end
                    if (master_read) begin
                        if (~prev_page_addr_valid | (prev_page_addr_valid & (master_address[24:20] != prev_page_addr))) begin
                            state                <= RD_WAIT_ACCEPT_P;
                            master_read_valid    <= 1'b0;
                            master_wait          <= 1'b1;
                            master_read_data     <= 32'd0;
                            slave_read           <= 1'b0;
                            slave_write          <= slave_enable ? 1'b1 : 1'b0;
                            slave_address        <= PAGE_ADDRESS;
                            slave_write_data     <= {3'b000,master_address[24:20]}; //Page Value
                            master_read_r        <= 1'b1;
                            master_write_r       <= 1'b0;
                            prev_page_addr       <= master_address[24:20];
                            prev_page_addr_valid <= 1'b1;
                        end
                        else begin
                            state                <= RD_WAIT_ACCEPT_0;
                            master_read_valid    <= 1'b0;
                            master_wait          <= 1'b1;
                            master_read_data     <= 32'd0;
                            slave_read           <= slave_enable ? 1'b1 : 1'b0;
                            slave_write          <= 1'b0;
                            slave_address        <= {12'h1,master_address[19:2], 2'd0};
                            slave_write_data     <= 8'd0;
                            master_read_r        <= 1'b1;
                            master_write_r       <= 1'b0;
                        end
                    end
                end
                WR_WAIT_P           : begin
                    if (!slave_wait) begin
                        casez (master_byte_enable)
                            4'b???1 : begin
                                state               <= WR_WAIT_0;
                                slave_address       <= {12'h1,master_address[19:2], 2'd0};
                                slave_write_data    <= master_write_data[7:0];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b??10 : begin
                                state               <= WR_WAIT_1;
                                slave_address       <= {12'h1,master_address[19:2], 2'd1};
                                slave_write_data    <= master_write_data[15:8];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b?100 : begin
                                state               <= WR_WAIT_2;
                                slave_address       <= {12'h1,master_address[19:2], 2'd2};
                                slave_write_data    <= master_write_data[23:16];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b1000 : begin
                                state               <= WR_WAIT_3;
                                slave_address       <= {12'h1,master_address[19:2], 2'd3};
                                slave_write_data    <= master_write_data[31:24];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b0000 : begin
                                state               <= WR_DONE;
                                master_wait         <= 1'b0;
                                slave_write         <= 1'b0;
                            end
                        endcase
                    end
                end
                WR_WAIT_0           : begin
                    if (!slave_wait) begin
                        casez (master_byte_enable)
                            4'b??1? : begin
                                state               <= WR_WAIT_1;
                                slave_address       <= {12'h1,master_address[19:2], 2'd1};
                                slave_write_data    <= master_write_data[15:8];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b?10? : begin
                                state               <= WR_WAIT_2;
                                slave_address       <= {12'h1,master_address[19:2], 2'd2};
                                slave_write_data    <= master_write_data[23:16];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b100? : begin
                                state               <= WR_WAIT_3;
                                slave_address       <= {12'h1,master_address[19:2], 2'd3};
                                slave_write_data    <= master_write_data[31:24];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b000? : begin
                                state               <= WR_DONE;
                                master_wait         <= 1'b0;
                                slave_write         <= 1'b0;
                            end
                        endcase
                    end
                end
                WR_WAIT_1           : begin
                    if (!slave_wait) begin
                        casez (master_byte_enable)
                            4'b?1?? : begin
                                state               <= WR_WAIT_2;
                                slave_address       <= {12'h1,master_address[19:2], 2'd2};
                                slave_write_data    <= master_write_data[23:16];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b10?? : begin
                                state               <= WR_WAIT_3;
                                slave_address       <= {12'h1,master_address[19:2], 2'd3};
                                slave_write_data    <= master_write_data[31:24];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b00?? : begin
                                state               <= WR_DONE;
                                master_wait         <= 1'b0;
                                slave_write         <= 1'b0;
                            end
                        endcase
                    end
                end
                WR_WAIT_2           : begin
                    if (!slave_wait) begin
                        casez (master_byte_enable)
                            4'b1??? : begin
                                state               <= WR_WAIT_3;
                                slave_address       <= {12'h1,master_address[19:2], 2'd3};
                                slave_write_data    <= master_write_data[31:24];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b0??? : begin
                                state               <= WR_DONE;
                                master_wait         <= 1'b0;
                                slave_write         <= 1'b0;
                            end
                        endcase
                    end
                end
                WR_WAIT_3           : begin
                    if (!slave_wait) begin
                        state               <= WR_DONE;
                        master_wait         <= 1'b0;
                        slave_write         <= 1'b0;
                    end
                end
                WR_DONE             : begin
                    state               <= IDLE;
                    master_wait         <= 1'b1;
                end
                RD_WAIT_ACCEPT_P    : begin
                    if (!slave_wait) begin
                        state               <= RD_WAIT_ACCEPT_0;
                        master_read_valid   <= 1'b0;
                        master_wait         <= 1'b1;
                        master_read_data    <= 32'd0;
                        slave_read          <= slave_enable ? 1'b1 : 1'b0;
                        slave_write         <= 1'b0;
                        slave_address       <= {12'h1,master_address[19:2], 2'd0};
                        slave_write_data    <= 8'd0;
                    end
                end
                RD_WAIT_ACCEPT_0    : begin
                    if (!slave_wait) begin
                        if (slave_read_valid & slave_enable) begin
                            state                   <= RD_WAIT_ACCEPT_1;
                            master_read_data[7:0]   <= slave_read_data;
                            slave_address           <= {12'h1,master_address[19:2], 2'd1};
                        end else begin
                            state                   <= RD_WAIT_VALID_0;
                            slave_read              <= 1'b0;
                        end
                    end
                end
                RD_WAIT_ACCEPT_1    : begin
                    if (!slave_wait) begin
                        if (slave_read_valid & slave_enable) begin
                            state                   <= RD_WAIT_ACCEPT_2;
                            master_read_data[15:8]  <= slave_read_data;
                            slave_address           <= {12'h1,master_address[19:2], 2'd2};
                        end else begin
                            state                   <= RD_WAIT_VALID_1;
                            slave_read              <= 1'b0;
                        end
                    end
                end
                RD_WAIT_ACCEPT_2    : begin
                    if (!slave_wait) begin
                        if (slave_read_valid & slave_enable) begin
                            state                   <= RD_WAIT_ACCEPT_3;
                            master_read_data[23:16] <= slave_read_data;
                            slave_address           <= {12'h1,master_address[19:2], 2'd3};
                        end else begin
                            state                   <= RD_WAIT_VALID_2;
                            slave_read              <= 1'b0;
                        end
                    end
                end
                RD_WAIT_ACCEPT_3    : begin
                    if (!slave_wait) begin
                        if (slave_read_valid & slave_enable) begin
                            state                   <= RD_VALID;
                            master_wait             <= 1'b0;
                            master_read_data[31:24] <= slave_read_data;
                            slave_read              <= 1'b0;
                            master_read_valid       <= 1'b1;
                        end else begin
                            state                   <= RD_WAIT_VALID_3;
                            slave_read              <= 1'b0;
                        end
                    end
                end
                RD_WAIT_VALID_0     : begin
                    if (slave_read_valid) begin
                        state                   <= RD_WAIT_ACCEPT_1;
                        master_read_data[7:0]   <= slave_read_data;
                        slave_address           <= {12'h1,master_address[19:2], 2'd1};
                        slave_read              <= slave_enable ? 1'b1 : 1'b0;
                    end
                end
                RD_WAIT_VALID_1     : begin
                    if (slave_read_valid) begin
                        state                   <= RD_WAIT_ACCEPT_2;
                        master_read_data[15:8]  <= slave_read_data;
                        slave_address           <= {12'h1,master_address[19:2], 2'd2};
                        slave_read              <= slave_enable ? 1'b1 : 1'b0;
                    end
                end
                RD_WAIT_VALID_2     : begin
                    if (slave_read_valid) begin
                        state                   <= RD_WAIT_ACCEPT_3;
                        master_read_data[23:16] <= slave_read_data;
                        slave_address           <= {12'h1,master_address[19:2], 2'd3};
                        slave_read              <= slave_enable ? 1'b1 : 1'b0;
                    end
                end
                RD_WAIT_VALID_3     : begin
                    if (slave_read_valid) begin
                        state                   <= RD_VALID;
                        master_wait             <= 1'b0;
                        master_read_data[31:24] <= slave_read_data;
                        slave_read              <= 1'b0;
                        master_read_valid       <= 1'b1;
                    end
                end
                RD_ACCEPT           : begin
                    state                       <= RD_VALID;
                    master_read_valid           <= 1'b1;
                    master_wait                 <= 1'b1;
                end
                RD_VALID            : begin
                    state                       <= IDLE;
                    master_read_valid           <= 1'b0;
                    master_wait                 <= 1'b1;
                end
                default             : begin
                    state               <= IDLE;
                    master_read_valid   <= 1'b0;
                    master_wait         <= 1'b1;
                    master_read_data    <= 32'd0;
                    slave_read          <= 1'b0;
                    slave_write         <= 1'b0;
                    slave_address       <= 32'd0;
                    slave_write_data    <= 8'd0;
                    master_write_r      <= 1'b0;
                    master_read_r       <= 1'b0;
                end
            endcase
        end
    end

`ifdef DR_AVMM_SIM_DEBUG_EN
// synthesis translate_off
    always @(posedge clk) begin
        if (((timer === avmm_timeout) & (|avmm_timeout)) === 1'b1) begin
            $display("%t %m: internal error, GAVMM access timeout", $time);
        end
    end
// synthesis translate_on

// synthesis translate_off
    always @(posedge clk) begin
        if (master_read_valid === 1'b1) begin
            $display("%t: internal info, GAVMM read  data = 0x%h from address = 0x%h", $time, master_read_data, master_address);
        end
    end

    always @(posedge clk) begin
        if ((master_write & (state === WR_DONE)) === 1'b1) begin
            $display("%t: internal info, GAVMM write data = 0x%h to   address = 0x%h", $time, master_write_data, master_address);
        end
    end
// synthesis translate_on
`endif

endmodule //DR_F_GAVMM_32to8
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "wuj6SUN+CWUHl7X/28Op4/b/clpjqbLY1GA2Z3hIF6AQ/N7zu6Fwoye5cSVZtCMjzq7zVIP70Vz25qwBBwoIUIbppOYuSWAZCOtwfVezNqtYswQ62n/3Hi6fWGVfeSbJS3aJs8eFyk/7NahU2OTeak7iTUPHQndN4oB5O1gSp87DXZC4zl72BeNdhBCD27/XMw5oUpTkQ1qs22Ma/P0Qj3KuL2yUnt0fdnpvUGYwRExO7Gsbuz6dvQQJTg07sVUjSPXh7iD8us0a5RQOaXfoUgpkkx4uSGFOalTRwyNdFieGoFVB/N3HI9qux9hPWVaAUXW3nBn9+7wIE/D2YItyucM4sf6Vr4Ol5UQ5pAKnWWXtHSpNtsU/qm3dNpPo1uIW+NUJXGnGDWzKZXE1pfckrdaVMdbok7P/pomX/U1QbmUDl0zYBiqSHJdud5n1fL860QD+5YR260P/Bow/CLTMy+9bTQzxOHJMYa0Jdr9KJQnx3pCjUrjfyB82wavVUz9pJbFMjCcGV/n9jtV2AtzpWMje72Nbxa0LzbON/Wl7ttyrnhpiBxcNRFnovqMS0UlbBa6qw/8oUxkqniKfUWlkfmxNjSjrOhZcLG2GYDurnThDqtQC8XkfbZTGmyTBpwJrQX6UcsCKIi9TvDAB8l3nn6RKC79kQPxKE0z9ICI22NAlqnLVWVO9GR8u22l5IXJY3VbLrEXkwkhfXWMUr+CeP0JXAELdYe8OUrGmtCMtZ9KnNVxTbAn/1zJ6GfK5ucmswHGXzoQ4nDVZL8D3dVzGHqMgA/q/y8U6oXRMwu/+XDkYDyV1l8+02tMzXG/Sz8H4rROgHTkL4EARbjcPNKQMpbU4Izt5S0NVz3kPvWBj7jpG/bIFwUdxuF0/aiNu4viuxuXn1plbY2NPQ3EBvjFHQFOpjYdRsMFkKhRCizfcTJe+OBWI8r6l83Tt0FOxew8m3qTIdc4EhNUXSo/yvYvmt5tXZ7dmKaeX3jkCh528/pLQuTwx5/aLifDuq0Brg/la"
`endif