// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


//----------------------------------------------------------------------------- 
//  Project Name:  GDR Dynamic Reconfiguration (F-tile)                                  
//  Module Name :  dr_f_avmm2mem_cc_bridge                                  
//  Author ID   :  hafeezmu                                   
//  Date        :  Wed July 29 11:27:46 2020   
//  Version     :  v1.0 (Initial Draft)                               
//  Description :  GDR DR AVMM to memory clock crossing bridge is used to     
//                 access the Soft reset controller memory address space.                                              
//-----------------------------------------------------------------------------  
// synthesis translate_off
`timescale 1 ps / 1 fs
// synthesis translate_on
module dr_f_avmm2mem_cc_bridge #(
        parameter DWIDTH        = 32,
        parameter AWIDTH        = 16,
        parameter DEPTH         = 3'd2  // valid value from 2 to 6
    ) (   
    input   logic                     i_avmm_clk,
    input   logic                     i_avmm_rst_n,
    input   logic  [DWIDTH-1:0]       i_avmm_writedata,
    input   logic                     i_avmm_read,
    input   logic  [AWIDTH-1:0]       i_avmm_address,
    input   logic                     i_avmm_write,
    output  logic  [DWIDTH-1:0]       o_avmm_readdata,
    output  logic                     o_avmm_waitrequest,
    output  logic                     o_avmm_readdata_valid,
    
    input   logic                     i_mem_clk,
    input   logic                     i_mem_rst,
    input   logic                     i_mem_grant,
    output  logic  [AWIDTH-1:0]       o_mem_addr,
    output  logic                     o_mem_we,
    output  logic  [DWIDTH-1:0]       o_mem_wdata,
    input   logic  [DWIDTH-1:0]       i_mem_rdata
    );
   

    // 1: Protocol AVMM management
    // combinatorial read data signal declaration
    logic               avmm_read_edge;
    logic               avmm_write_edge;
    logic  [2:0]        avmm_write_cnt;
    logic               avmm_write_valid;
    logic               avmm_read_valid;
    logic               read_mem_valid;
    logic  [5:0]        write_mem_valid;
    logic               write_mem_request;
    logic  [DWIDTH-1:0] mem_wdata;
    logic  [AWIDTH-1:0] mem_addr;    
    logic  [DWIDTH-1:0] avmm_readdata;
    logic               avmm_write_ready;
    logic               avmm_write_ready_r;

    // synchronous process for the read
    always @(posedge i_avmm_clk or negedge i_avmm_rst_n) begin
       if (!i_avmm_rst_n) 
          o_avmm_readdata <= {DWIDTH{1'b0}};
       else 
          o_avmm_readdata <= avmm_read_valid ? avmm_readdata : {DWIDTH{1'b0}};
    end
    
    // read data is always returned on the next cycle
    always @(posedge i_avmm_clk or negedge i_avmm_rst_n) begin
       if (!i_avmm_rst_n) 
         o_avmm_readdata_valid <= 1'b0; 
       else 
         o_avmm_readdata_valid <= avmm_read_valid;
    end
    
    always @(posedge i_avmm_clk or negedge i_avmm_rst_n) begin
       if (!i_avmm_rst_n) 
         avmm_read_edge <= 1'b0; 
       else 
         avmm_read_edge <= i_avmm_read;
    end
    
    always @(posedge i_avmm_clk or negedge i_avmm_rst_n) begin
       if (!i_avmm_rst_n) 
         avmm_write_edge <= 1'b0; 
       else 
         avmm_write_edge <= i_avmm_write;
    end
    
    always @(posedge i_avmm_clk or negedge i_avmm_rst_n) begin
       if (!i_avmm_rst_n) 
         avmm_write_cnt <= DEPTH; 
       else if (i_avmm_write & (|avmm_write_cnt))
         avmm_write_cnt <= avmm_write_cnt - 1'b1;
       else if (~i_avmm_write)
         avmm_write_cnt <= DEPTH; 
    end

    always @(posedge i_avmm_clk or negedge i_avmm_rst_n) begin
       if (!i_avmm_rst_n) 
         avmm_write_ready_r <= 1'b0; 
       else
         avmm_write_ready_r <= avmm_write_ready; 
    end
    
    //assign o_avmm_waitrequest = (i_avmm_write & (|avmm_write_cnt)) | (i_avmm_read & (~o_avmm_readdata_valid));
    
    assign o_avmm_waitrequest = (i_avmm_write & ((|avmm_write_cnt) | ~avmm_write_ready_r)) | (i_avmm_read & (~o_avmm_readdata_valid));

    //
    // 2: Protocol Clock Crossing specific assignment to inside signals
    //   
    assign avmm_write_valid = (i_avmm_write & ~avmm_write_edge) | (i_avmm_read & ~avmm_read_edge);
    
    always @(posedge i_mem_clk or posedge i_mem_rst) begin
       if (i_mem_rst)
         write_mem_valid <= 6'h0;
       else
         write_mem_valid <= {write_mem_valid[4:0], (~write_mem_request & read_mem_valid)};
    end         
       
    dr_f_clock_crosser #(
        .DATA_WIDTH             (AWIDTH+DWIDTH+1),
        .FORWARD_SYNC_DEPTH     (DEPTH),
        .BACKWARD_SYNC_DEPTH    (DEPTH)
    ) dr_src_avmm2mem_wr_req_cc (
        .in_clk                 (i_avmm_clk),
        .in_reset               (!i_avmm_rst_n),
        .in_ready               (avmm_write_ready),
        .in_valid               (avmm_write_valid),
        .in_data                ({i_avmm_write,i_avmm_writedata,i_avmm_address}),
        .out_clk                (i_mem_clk),
        .out_reset              (i_mem_rst),
        .out_ready              (1'b1),
        .out_valid              (read_mem_valid),
        .out_data               ({write_mem_request,mem_wdata,mem_addr})
    );

    dr_f_clock_crosser #(
        .DATA_WIDTH             (DWIDTH),
        .FORWARD_SYNC_DEPTH     (DEPTH),
        .BACKWARD_SYNC_DEPTH    (DEPTH)
    ) dr_src_avmm2mem_rd_req_cc (
        .in_clk                 (i_mem_clk),
        .in_reset               (i_mem_rst),
        .in_ready               (),
        .in_valid               (write_mem_valid[5]),
        .in_data                (i_mem_rdata),
        .out_clk                (i_avmm_clk),
        .out_reset              (!i_avmm_rst_n),
        .out_ready              (1'b1),
        .out_valid              (avmm_read_valid),
        .out_data               (avmm_readdata)
    );

    // 3: Protocol Memory management
    
    always @(posedge i_mem_clk or posedge i_mem_rst) begin
       if (i_mem_rst) begin
         o_mem_we    <= 1'b0;
         o_mem_wdata <= {DWIDTH{1'b0}};
         o_mem_addr  <= {AWIDTH{1'b0}};
       end
       else begin
         o_mem_we    <= (write_mem_request & read_mem_valid & i_mem_grant) ? 1'b1      : 1'b0;
         o_mem_wdata <= (write_mem_request & read_mem_valid)               ? mem_wdata : {DWIDTH{1'b0}};
         o_mem_addr  <= (read_mem_valid)                                   ? mem_addr  : o_mem_addr;
       end
    end   

`ifdef DR_AVMM_SIM_DEBUG_EN
// synthesis translate_off
    always @(posedge i_avmm_clk) begin
        if (o_avmm_readdata_valid === 1'b1) begin
            $display("%t: internal info, SRC read  data = 0x%h from memory address = 0x%h", $time, o_avmm_readdata, i_avmm_address);
        end
    end
    
    always @(posedge i_mem_clk) begin
        if (o_mem_we === 1'b1) begin
            $display("%t: internal info, SRC write data = 0x%h to   memory address = 0x%h", $time, o_mem_wdata, o_mem_addr);
        end
    end
// synthesis translate_on
`endif

endmodule //DR_F_AVMM2MEM_CC_BRIDGE
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "79ztKTnV2ny/eHpgZYepe9G9wwwdeg9hTOdIyLdh4JTuZZ1DT50cJOPNpZ0w+c5aVHkra3Tx9+PMlDB4omAPannbtZoJG9ZekMCcgtPHzQ4GWiJfmhYoHZqap4yb9/uKlHSl9TpAcV1FKEwkrctIKziGBKSQ0S70p/jW9wD4CtM3X7P+ggV0x7Prg108lDSsdB1kGvgcQN/5kLCIPO6i12claBXdWuhbdU7Bm3JHKTqZcO9lbZ5dXQ3YXc/udXJkNZjV2d5uKHRdwY7bBPiYKT2b6x5h45XnjKeaF8DSAiQXcMUAEKZ4hYxo25OI24+iLsWCu5C4ZB5bg2FrSpRhYKqO9KBbrKIKBpTffQRQ6hfZXsHnszvAlHTSeIb8ANfwZBT6sFV+2o2pFgYgQKyMzjC2oKHFir4xEamxq1hPnunhzEEtZGVA0PasMNQztHcA1EfHurqUKA4T9QSAGYVAhZhsKRhtHa+U/f3AR7LX4PKSshyoZo2t579u/5dD0+dtltar/lsEQLqxFjupWirTJoGCfcA8pQUSZ5oR2qRc7fa4yCvCh6aoJwiUGfhQobT73XzKhnLJ8WksPjNSljtV0Dm4zqU/IDJVPw0DPt/lJsur5iAOv9BqDy9OkvdFEl8hG1u9ilW9qNm053gGvessWT1TMa35Z/cfpPpmfMwsu6GIwVyW9ppJ+JfJzeqhuQmWNadcTOBMuPLd3V/EkXyY9+6nCNzZAF+4muStddg48MYpzD6Pkjk9AVwbOKZ3Nc//MadQNPq3CrsPr8V36h+ZaL3PPUysfLhUp92QM5cyoK1PKyMelj0E+h+9iP3//6ifQANCmveZIa9qnyV9HjzKoFLFxts5+BpyPJxEj4mnIzni00rp4MeejBeoTYoNjZoxM68wu1FPrsVY0ulmm8ZS0pN6w/EBnjOm3zXTZcycxdSLZoZzvD4YtH/c1TFtDjKQmN68tdhNDDB5V7116YdQM/yG3T288qS9yQqUJY/YIMl+6yDry0gxu1CRtXjfwokt"
`endif