// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


//----------------------------------------------------------------------------- 
//  Project Name:  GDR Dynamic Reconfiguration (F-tile)                                  
//  Module Name :  dr_f_avmm_32to8                                  
//  Author ID   :  hafeezmu                                   
//  Date        :  Fri Mar 06 10:17:34 2020   
//  Version     :  v1.0 (Initial Draft)                               
//  Description :  GDR DR 32bits to 8bits AVMM data conversion block for F-tile. 
//                 This component is top block of DR SIP that covert the Global
//                 AVMM accesses from NIOS 32bits to Hard IP 8bits                                            
//-----------------------------------------------------------------------------
// synthesis translate_off
`timescale 1 ps / 1 fs
// synthesis translate_on

module dr_f_avmm_32to8 #(
    parameter TIMEOUT = 1000
) (
    input               clk,
    input               rst_n,
    input       [31:0]  avmm_timeout, 
    output reg          avmm_error,
    
    input               master_read,
    input               master_write,
    input       [31:0]  master_address,
    input       [31:0]  master_write_data,
    input       [3:0]   master_byte_enable,
    output reg          master_read_valid,
    output reg          master_wait,
    output reg  [31:0]  master_read_data,

    input               slave_enable,
    output reg          slave_read,
    output reg          slave_write,
    output reg  [31:0]  slave_address,
    output reg  [ 7:0]  slave_write_data,
    input               slave_read_valid,
    input               slave_wait,
    input       [ 7:0]  slave_read_data
);

    enum {  IDLE,
            WR_WAIT_0 , WR_WAIT_1 , WR_WAIT_2 , WR_WAIT_3 , WR_DONE,
            RD_WAIT_ACCEPT_0 , RD_WAIT_ACCEPT_1 , RD_WAIT_ACCEPT_2 , RD_WAIT_ACCEPT_3,
            RD_WAIT_VALID_0 , RD_WAIT_VALID_1 , RD_WAIT_VALID_2 , RD_WAIT_VALID_3,
            RD_ACCEPT , RD_VALID } state;

    localparam TIMER_BITS = $clog2(TIMEOUT + 1);

    reg [31:0]              timer;
    reg                     slave_enable_r;
    wire                    slave_enable_redge;
    reg                     master_write_r;
    reg                     master_read_r;

    always_ff @(posedge clk) begin
        if (~rst_n) begin
            slave_enable_r   <= 1'b0;
        end else begin
            slave_enable_r   <= slave_enable;
        end
    end

    always_ff @(posedge clk) begin
        if (~rst_n) begin
            avmm_error <= 1'b0;
        end 
        else begin
            if ((timer == avmm_timeout) & (|avmm_timeout)) begin
                avmm_error <= 1'b1;
            end
            else begin
                avmm_error <= 1'b0;
            end
        end
    end
    
    assign slave_enable_redge = slave_enable & ~slave_enable_r;

    always_ff @(posedge clk) begin
        if (~rst_n) begin
            state               <= IDLE;
            master_read_valid   <= 1'b0;
            master_wait         <= 1'b1;
            master_read_data    <= 32'd0;
            slave_read          <= 1'b0;
            slave_write         <= 1'b0;
            slave_address       <= 32'd0;
            slave_write_data    <= 8'd0;
            timer               <= {32{1'b0}};
            master_write_r      <= 1'b0;
            master_read_r       <= 1'b0;
        end else begin
            state               <= state;
            master_read_valid   <= master_read_valid;
            master_wait         <= master_wait;
            master_read_data    <= master_read_data;
            slave_read          <= slave_read;
            slave_write         <= slave_write;
            slave_address       <= slave_address;
            slave_write_data    <= slave_write_data;
            master_write_r      <= master_write_r;
            master_read_r       <= master_read_r;

            // Timeout handler
            if ((state != IDLE) && ~slave_enable_redge) begin
                timer   <= timer + 1'b1;
                if ((timer == avmm_timeout) & (|avmm_timeout)) begin
                    master_wait         <= 1'b0;
                    slave_write         <= 1'b0;
                    slave_read          <= 1'b0;
                    if (master_write_r) begin
                        state               <= WR_DONE;
                    end else if (master_read_r) begin
                        state               <= RD_ACCEPT;
                    end
                end
            end else begin
                timer   <= {TIMER_BITS{1'b0}};
            end

            case (state)
                IDLE                : begin
                    if (master_write) begin
                        master_read_valid   <= 1'b0;
                        master_wait         <= 1'b1;
                        master_read_data    <= 32'd0;
                        slave_read          <= 1'b0;
                        master_write_r      <= 1'b1;
                        master_read_r       <= 1'b0;
                        casez (master_byte_enable)
                            4'b???1 : begin
                                state               <= WR_WAIT_0;
                                slave_address       <= {master_address[31:2], 2'd0};
                                slave_write_data    <= master_write_data[7:0];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b??10 : begin
                                state               <= WR_WAIT_1;
                                slave_address       <= {master_address[31:2], 2'd1};
                                slave_write_data    <= master_write_data[15:8];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b?100 : begin
                                state               <= WR_WAIT_2;
                                slave_address       <= {master_address[31:2], 2'd2};
                                slave_write_data    <= master_write_data[23:16];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b1000 : begin
                                state               <= WR_WAIT_3;
                                slave_address       <= {master_address[31:2], 2'd3};
                                slave_write_data    <= master_write_data[31:24];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b0000 : begin
                                state               <= WR_DONE;
                                master_wait         <= 1'b0;
                                slave_write         <= 1'b0;
                            end
                        endcase
                    end
                    if (master_read) begin
                        state               <= RD_WAIT_ACCEPT_0;
                        master_read_valid   <= 1'b0;
                        master_wait         <= 1'b1;
                        master_read_data    <= 32'd0;
                        slave_read          <= slave_enable ? 1'b1 : 1'b0;
                        slave_write         <= 1'b0;
                        slave_address       <= {master_address[31:2], 2'd0};
                        slave_write_data    <= 8'd0;
                        master_read_r       <= 1'b1;
                        master_write_r      <= 1'b0;
                    end
                end
                WR_WAIT_0           : begin
                    if (!slave_wait) begin
                        casez (master_byte_enable)
                            4'b??1? : begin
                                if (slave_write == 1'b1) begin
                                  slave_write       <= 1'b0;                               
                                end
                                else begin
                                  state             <= WR_WAIT_1;
                                  slave_address     <= {master_address[31:2], 2'd1};
                                  slave_write_data  <= master_write_data[15:8];
                                  slave_write       <= slave_enable ? 1'b1 : 1'b0;
                                end 
                            end
                            4'b?10? : begin
                                state               <= WR_WAIT_2;
                                slave_address       <= {master_address[31:2], 2'd2};
                                slave_write_data    <= master_write_data[23:16];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b100? : begin
                                state               <= WR_WAIT_3;
                                slave_address       <= {master_address[31:2], 2'd3};
                                slave_write_data    <= master_write_data[31:24];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b000? : begin
                                state               <= WR_DONE;
                                master_wait         <= 1'b0;
                                slave_write         <= 1'b0;
                            end
                        endcase
                    end
                end
                WR_WAIT_1           : begin
                    if (!slave_wait) begin
                        casez (master_byte_enable)
                            4'b?1?? : begin
                                if (slave_write == 1'b1) begin
                                  slave_write       <= 1'b0;                               
                                end
                                else begin
                                  state             <= WR_WAIT_2;
                                  slave_address     <= {master_address[31:2], 2'd2};
                                  slave_write_data  <= master_write_data[23:16];
                                  slave_write       <= slave_enable ? 1'b1 : 1'b0;
                                end
                            end
                            4'b10?? : begin
                                state               <= WR_WAIT_3;
                                slave_address       <= {master_address[31:2], 2'd3};
                                slave_write_data    <= master_write_data[31:24];
                                slave_write         <= slave_enable ? 1'b1 : 1'b0;
                            end
                            4'b00?? : begin
                                state               <= WR_DONE;
                                master_wait         <= 1'b0;
                                slave_write         <= 1'b0;
                            end
                        endcase
                    end
                end
                WR_WAIT_2           : begin
                    if (!slave_wait) begin
                        casez (master_byte_enable)
                            4'b1??? : begin
                                if (slave_write == 1'b1) begin
                                  slave_write       <= 1'b0;                               
                                end
                                else begin
                                  state             <= WR_WAIT_3;
                                  slave_address     <= {master_address[31:2], 2'd3};
                                  slave_write_data  <= master_write_data[31:24];
                                  slave_write       <= slave_enable ? 1'b1 : 1'b0;
                                end
                            end
                            4'b0??? : begin
                                state               <= WR_DONE;
                                master_wait         <= 1'b0;
                                slave_write         <= 1'b0;
                            end
                        endcase
                    end
                end
                WR_WAIT_3           : begin
                    if (!slave_wait) begin
                        state               <= WR_DONE;
                        master_wait         <= 1'b0;
                        slave_write         <= 1'b0;
                    end
                end
                WR_DONE             : begin
                    state               <= IDLE;
                    master_wait         <= 1'b1;
                end
                RD_WAIT_ACCEPT_0    : begin
                    if (!slave_wait) begin
                        if (slave_read_valid & slave_enable) begin
                            state                   <= RD_WAIT_ACCEPT_1;
                            master_read_data[7:0]   <= slave_read_data;
                            slave_address           <= {master_address[31:2], 2'd1};
                        end else begin
                            state                   <= RD_WAIT_VALID_0;
                            slave_read              <= 1'b0;
                        end
                    end
                end
                RD_WAIT_ACCEPT_1    : begin
                    if (!slave_wait) begin
                        if (slave_read_valid & slave_enable) begin
                            state                   <= RD_WAIT_ACCEPT_2;
                            master_read_data[15:8]  <= slave_read_data;
                            slave_address           <= {master_address[31:2], 2'd2};
                        end else begin
                            state                   <= RD_WAIT_VALID_1;
                            slave_read              <= 1'b0;
                        end
                    end
                end
                RD_WAIT_ACCEPT_2    : begin
                    if (!slave_wait) begin
                        if (slave_read_valid & slave_enable) begin
                            state                   <= RD_WAIT_ACCEPT_3;
                            master_read_data[23:16] <= slave_read_data;
                            slave_address           <= {master_address[31:2], 2'd3};
                        end else begin
                            state                   <= RD_WAIT_VALID_2;
                            slave_read              <= 1'b0;
                        end
                    end
                end
                RD_WAIT_ACCEPT_3    : begin
                    if (!slave_wait) begin
                        if (slave_read_valid & slave_enable) begin
                            state                   <= RD_VALID;
                            master_wait             <= 1'b0;
                            master_read_data[31:24] <= slave_read_data;
                            slave_read              <= 1'b0;
                            master_read_valid       <= 1'b1;
                        end else begin
                            state                   <= RD_WAIT_VALID_3;
                            slave_read              <= 1'b0;
                        end
                    end
                end
                RD_WAIT_VALID_0     : begin
                    if (slave_read_valid) begin
                        state                   <= RD_WAIT_ACCEPT_1;
                        master_read_data[7:0]   <= slave_read_data;
                        slave_address           <= {master_address[31:2], 2'd1};
                        slave_read              <= slave_enable ? 1'b1 : 1'b0;
                    end
                end
                RD_WAIT_VALID_1     : begin
                    if (slave_read_valid) begin
                        state                   <= RD_WAIT_ACCEPT_2;
                        master_read_data[15:8]  <= slave_read_data;
                        slave_address           <= {master_address[31:2], 2'd2};
                        slave_read              <= slave_enable ? 1'b1 : 1'b0;
                    end
                end
                RD_WAIT_VALID_2     : begin
                    if (slave_read_valid) begin
                        state                   <= RD_WAIT_ACCEPT_3;
                        master_read_data[23:16] <= slave_read_data;
                        slave_address           <= {master_address[31:2], 2'd3};
                        slave_read              <= slave_enable ? 1'b1 : 1'b0;
                    end
                end
                RD_WAIT_VALID_3     : begin
                    if (slave_read_valid) begin
                        state                   <= RD_VALID;
                        master_wait             <= 1'b0;
                        master_read_data[31:24] <= slave_read_data;
                        slave_read              <= 1'b0;
                        master_read_valid       <= 1'b1;
                    end
                end
                RD_ACCEPT           : begin
                    state                       <= RD_VALID;
                    master_read_valid           <= 1'b1;
                    master_wait                 <= 1'b1;
                end
                RD_VALID            : begin
                    state                       <= IDLE;
                    master_read_valid           <= 1'b0;
                    master_wait                 <= 1'b1;
                end
                default             : begin
                    state               <= IDLE;
                    master_read_valid   <= 1'b0;
                    master_wait         <= 1'b1;
                    master_read_data    <= 32'd0;
                    slave_read          <= 1'b0;
                    slave_write         <= 1'b0;
                    slave_address       <= 32'd0;
                    slave_write_data    <= 8'd0;
                    master_write_r      <= 1'b0;
                    master_read_r       <= 1'b0;
                end
            endcase
        end
    end
    
`ifdef DR_AVMM_SIM_DEBUG_EN
// synthesis translate_off
    always @(posedge clk) begin
        if (((timer === avmm_timeout) & (|avmm_timeout)) === 1'b1) begin
            $display("%t %m: internal error, MAIB AVMM1 access timeout", $time);
        end
    end
// synthesis translate_on

// synthesis translate_off
    always @(posedge clk) begin
        if (master_read_valid === 1'b1) begin
            $display("%t: internal info, MAIB AVMM1 read  data = 0x%h from address = 0x%h", $time, master_read_data, master_address);
        end
    end

    always @(posedge clk) begin
        if ((master_write & ~master_write_r) === 1'b1) begin
            $display("%t: internal info, MAIB AVMM1 write data = 0x%h to   address = 0x%h", $time, master_write_data, master_address);
        end
    end
// synthesis translate_on
`endif

endmodule //DR_F_AVMM_32to8
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "79ztKTnV2ny/eHpgZYepe9G9wwwdeg9hTOdIyLdh4JTuZZ1DT50cJOPNpZ0w+c5aVHkra3Tx9+PMlDB4omAPannbtZoJG9ZekMCcgtPHzQ4GWiJfmhYoHZqap4yb9/uKlHSl9TpAcV1FKEwkrctIKziGBKSQ0S70p/jW9wD4CtM3X7P+ggV0x7Prg108lDSsdB1kGvgcQN/5kLCIPO6i12claBXdWuhbdU7Bm3JHKTr+0wsU7h20Lym5Hvvm9+K/TTDf872m83rDgnImrIEz2Z2DwNgpUx/GB1AQvxX0T/O1q/u+4Ow64AK5lMyR8otYwplGxxoZGWepmSGa5kVVXzhQ6NLYKeHCkEC+FI5/X9Lp9npAjkX6oxIXGZ41K5nncD2/ISj7YjXoSffQZR6ihAOyEjtytjIzQr5/dNtOwik6nugevg5Cf/AdLnIT6qkoEL6x6LmCME+uVEMmHawBUvB8myAqYJoQeWPOkBc7bL0Sc0b2OKXCerX47nhL2Z258i+G5WyWnHNPVs+9vpZJZjk5I4SwAgXZZfK8enF6qyRMhlkYPHOupjA9u4zEQbHCptSHNIJ0pWVTahSNzywtjD1fw8pGoSuL93dG9djYvnwlfKiwsRLoSaERQMA3bbnpookRIVvjNbpkpvMYBmY5hCCsG9GgQ/bS446ZpGbpFv/1HS/ewJnya+Lz2zFcJRSxM8dihxS3hd2Xrx/EGvaTzScJtXkMEvBkJUzzNLdWDth1D+VNgkhwNVOE1sJP57ogBi7E+KVQCXeoTr4hGPLgeYykpsH6Gie8E5iJTKQDvro2B9Q82xdRME8XNqCrNoWeGFnUWDNaCK/nl1qSHHdnM391nOlYNhiXXx1Ud/Q7BvjiGbxSpOD39ak7n7bQYCVGLEXdkw/b2FgEpbL8z0OJwkgwaJllQ1l43nv4lYToeZnfDNlN5slPRHqdrXFLjpCFqsZA0GaQJWpuMXvImMXL5qk+ma3J7OXozJkAS4+LBg1xCWImd79snnliw4Z9YawG"
`endif