// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.



//----------------------------------------------------------------------------- 
//  Project Name:  GDR Dynamic Reconfiguration (F-tile)                                  
//  Module Name :  dr_f_avmm_interconnect                                  
//  Author ID   :  hafeezmu                                   
//  Date        :  Wed Feb 19 10:27:00 2020   
//  Version     :  v1.0 (Initial Draft)                               
//  Description :  GDR DR AVMM interconnect is to split single AVMM to multiple    
//                 AVMM slave interfaces. This component is part of DR SIP.This  
//                 commponent use to access local AVMM1 24 channels, Global AVMM
//                 and DR CSR address space                                             
//-----------------------------------------------------------------------------  

module dr_f_avmm_interconnect #(
    parameter NUM_SLAVE = 4,
    parameter TIMEOUT = 6000, 
    parameter SLAVE0_BASE_ADDR = ('h00_00_0000), //DR CSR
    parameter SLAVE1_BASE_ADDR = ('h02_00_0000), //GAVMM
    parameter SLAVE2_BASE_ADDR = ('h01_00_0000), //SRC
    parameter SLAVE3_BASE_ADDR = ('h00_80_0000), //MAIB
    parameter SLAVE4_BASE_ADDR = ('h08_00_0000),
    parameter SLAVE5_BASE_ADDR = ('h10_00_0000)
) 
(                          
    input   logic                      clk,
    input   logic                      rst_n,
    input   logic [31:0]               avmm_timeout,
    // master port
    input   logic [31:0]               avmm_mst_address,
    input   logic                      avmm_mst_read,
    input   logic                      avmm_mst_write,
    input   logic [31:0]               avmm_mst_writedata,
    output  logic [31:0]               avmm_mst_readdata,
    output  logic                      avmm_mst_readdata_valid,
    output  logic                      avmm_mst_waitrequest,
    // slave ports
    output  logic [31:0]               avmm_slv_address,
    output  logic [NUM_SLAVE-1:0]      avmm_slv_read,
    output  logic [NUM_SLAVE-1:0]      avmm_slv_write,
    output  logic [31:0]               avmm_slv_writedata,
    input   logic [(NUM_SLAVE*32)-1:0] avmm_slv_readdata,
    input   logic [NUM_SLAVE-1:0]      avmm_slv_readdata_valid,
    input   logic [NUM_SLAVE-1:0]      avmm_slv_waitrequest,
    output  logic [NUM_SLAVE-1:0]      slave_sel
);

`ifdef AVMM_SPLITTER_BYPASS
   always @(*) begin
      slave_sel                = {5'b0, 1'b1};// always select default slave
      avmm_slv_address         = avmm_mst_address;
      avmm_slv_read            = {5'b0, avmm_mst_read};
      avmm_slv_write           = {5'b0, avmm_mst_write};
      avmm_slv_writedata       = avmm_mst_writedata;
      avmm_mst_readdata        = avmm_slv_readdata[31:0];
      avmm_mst_readdata_valid  = avmm_slv_readdata_valid[0];
      avmm_mst_waitrequest     = avmm_slv_waitrequest[0];
   end
`else
   localparam STATE_IDLE  = 2'b00;
   localparam STATE_READ  = 2'b01;
   localparam STATE_WRITE = 2'b10;
   localparam STATE_WAIT  = 2'b11;

   reg [1:0] present_state, next_state;
   reg [31:0] address_reg;
   reg [31:0] writedata_reg;
   reg 	      waitrequest_r;
   reg        avmm_mst_write_r;
   reg        avmm_slv_waitrequest_r_slv_0;
   wire       waitrequest_w;
   wire       readdata_valid_w;
   wire       waitrequest_read_pulse_slv_0;
   
   localparam TIMER_BITS = $clog2(TIMEOUT + 1);
   reg [31:0]   timer;
   reg 			time_out;

   // Timeout handler and calculation 
   always @(posedge clk)
     begin
	if (!rst_n) begin
	   timer    <= {32{1'b0}};
	   time_out <= 1'b1;
	end 
	else if ((present_state == STATE_READ) || (present_state == STATE_WRITE) 
                 || (present_state == STATE_WAIT)) begin
	   timer   <= timer + 1'b1;
	   if ((timer == (avmm_timeout + 10'd500)) & (|avmm_timeout)) begin
              time_out <= 1'b1;
	   end
	end 
	else begin
	   time_out <= 1'b0;
	   timer    <= {TIMER_BITS{1'b0}};
	end
     end

   generate
   if (NUM_SLAVE == 4) begin: SLAVE_SEL_4 
      always @(posedge clk)
      begin
	  if (!rst_n)
	     slave_sel <= {NUM_SLAVE{1'b0}};
	  else if (present_state == STATE_IDLE) begin
	     if (avmm_mst_address[25:23] == SLAVE3_BASE_ADDR[25:23])
                slave_sel <= {1'b1, 1'b0, 1'b0, 1'b0};
	     else if (avmm_mst_address[25:16] == SLAVE2_BASE_ADDR[25:16])
                slave_sel <= {1'b0, 1'b1, 1'b0, 1'b0};
	     else if (avmm_mst_address[25] == SLAVE1_BASE_ADDR[25])
                slave_sel <= {1'b0, 1'b0, 1'b1, 1'b0};
	     else if (avmm_mst_address[25:23] == SLAVE0_BASE_ADDR[25:23])
                slave_sel <= {1'b0, 1'b0, 1'b0, 1'b1};
	     else // default slave
                slave_sel <= {1'b0, 1'b0, 1'b0, 1'b1};
	  end
          else if (next_state == STATE_IDLE)
             slave_sel <= {NUM_SLAVE{1'b0}};
	  else
	     slave_sel <= slave_sel;
      end   
   end: SLAVE_SEL_4
   else begin: SLAVE_SEL_NON_4
      always @(posedge clk)
      begin
	  if (!rst_n)
	     slave_sel <= {NUM_SLAVE{1'b0}};
	  else if (present_state == STATE_IDLE) begin
	     if (avmm_mst_address[31:25] == SLAVE5_BASE_ADDR[31:25])
	        slave_sel <= {1'b1, 1'b0,1'b0, 1'b0, 1'b0, 1'b0};
	     else if (avmm_mst_address[31:25] == SLAVE4_BASE_ADDR[31:25])
	        slave_sel <= {1'b0, 1'b1,1'b0, 1'b0, 1'b0, 1'b0};
	     else if (avmm_mst_address[31:25] == SLAVE3_BASE_ADDR[31:25])
                slave_sel <= {1'b0, 1'b0 ,1'b1, 1'b0, 1'b0, 1'b0};
	     else if (avmm_mst_address[31:25] == SLAVE2_BASE_ADDR[31:25])
                slave_sel <= {1'b0, 1'b0,1'b0, 1'b1, 1'b0, 1'b0};
	     else if (avmm_mst_address[31:25] == SLAVE1_BASE_ADDR[31:25])
                slave_sel <= {1'b0, 1'b0,1'b0, 1'b0, 1'b1, 1'b0};
	     else if (avmm_mst_address[31:25] == SLAVE0_BASE_ADDR[31:25])
                slave_sel <= {1'b0, 1'b0,1'b0, 1'b0, 1'b0, 1'b1};
	     else // default slave
                slave_sel <= {1'b0, 1'b0,1'b0, 1'b0, 1'b0, 1'b1};
	  end
          else if (next_state == STATE_IDLE)
             slave_sel <= {NUM_SLAVE{1'b0}};
	  else
	     slave_sel <= slave_sel;
      end
    end: SLAVE_SEL_NON_4
   endgenerate
    
   always @(posedge clk)
     begin
	if (!rst_n)
	  address_reg <= {32{1'b0}};
	else if (present_state == STATE_IDLE)
	  address_reg <= avmm_mst_address;
	else
	  address_reg <= address_reg;
    end

   always @(posedge clk)
     begin
	if (!rst_n)
	  writedata_reg <= {32{1'b0}};
	else if (present_state == STATE_IDLE)
	  writedata_reg <= avmm_mst_writedata;
	else
	  writedata_reg <= writedata_reg;
    end

   always @(posedge clk)
     begin
	if (!rst_n)
	  avmm_slv_waitrequest_r_slv_0 <= 1'b0;
	else
	  avmm_slv_waitrequest_r_slv_0 <= avmm_slv_waitrequest[0];
    end

   assign waitrequest_read_pulse_slv_0 = (slave_sel[0] & avmm_slv_waitrequest_r_slv_0 & ~avmm_slv_waitrequest[0] & avmm_mst_read);
   
   assign waitrequest_w = ((avmm_mst_read | avmm_mst_write) & |(avmm_slv_waitrequest & slave_sel));
   assign readdata_valid_w = |(avmm_slv_readdata_valid & slave_sel);

   always @(posedge clk)
   begin
    if (!rst_n)
    begin
      waitrequest_r <= 1'b1;
      avmm_mst_write_r <= 1'b0;
    end
    else
    begin
      waitrequest_r <= waitrequest_w;
      avmm_mst_write_r <= avmm_mst_waitrequest & avmm_mst_write;
    end
   end
   
   
   always @(posedge clk)
     begin
	if (!rst_n)
	  present_state <= STATE_IDLE;
	else
	  present_state <= next_state;
     end

   always @(*)
     begin
	case (present_state)
	  STATE_IDLE:  begin
             if (waitrequest_r)
               next_state = present_state;
             else if (avmm_mst_read)
               next_state = STATE_READ;
             else if (avmm_mst_write & ~avmm_mst_write_r)
               next_state = STATE_WRITE;
             else
               next_state = present_state;
          end
	  STATE_READ:  begin
             if (((!waitrequest_w & readdata_valid_w) | time_out))
               next_state = STATE_IDLE;
             else if ((!waitrequest_w & !readdata_valid_w))
               next_state = STATE_WAIT;
             else
               next_state = present_state;
          end
	  STATE_WRITE: begin
             if (!waitrequest_w | time_out)
               next_state = STATE_IDLE;
             else
               next_state = present_state;
          end
	  STATE_WAIT:  begin
             if (readdata_valid_w | time_out)
               next_state = STATE_IDLE;
             else
               next_state = present_state;
          end
	endcase
     end

   always @(*)
     begin
	case (present_state)
	  STATE_IDLE:  begin
             avmm_slv_address = avmm_mst_address;
             avmm_slv_writedata = avmm_mst_writedata;
             avmm_slv_write = {NUM_SLAVE{avmm_mst_write}} & slave_sel;
             avmm_slv_read = {NUM_SLAVE{avmm_mst_read}} & slave_sel;
             avmm_mst_waitrequest = 1'b1;
          end
	  STATE_READ:  begin
             avmm_slv_address = address_reg; 
             avmm_slv_writedata = writedata_reg;
             avmm_slv_write = {NUM_SLAVE{1'b0}};
             avmm_slv_read = slave_sel;
             avmm_mst_waitrequest = time_out ? 1'b0 : (waitrequest_read_pulse_slv_0 | waitrequest_w);
          end
	  STATE_WRITE: begin
             avmm_slv_address = address_reg; 
             avmm_slv_writedata = writedata_reg;
             avmm_slv_write = slave_sel;
             avmm_slv_read = {NUM_SLAVE{1'b0}};
             avmm_mst_waitrequest = time_out ? 1'b0 : waitrequest_w;
          end
	  STATE_WAIT:  begin
             avmm_slv_address = address_reg; 
             avmm_slv_writedata = writedata_reg;
             avmm_slv_write = {NUM_SLAVE{1'b0}};
             avmm_slv_read = {NUM_SLAVE{1'b0}};
             avmm_mst_waitrequest = 1'b0; //time_out ? 1'b0 : waitrequest_w;
          end
	endcase
     end

   always @(posedge clk)
   begin
     if (!rst_n)
       avmm_mst_readdata_valid <= 1'b0;
     else 
       avmm_mst_readdata_valid <= readdata_valid_w;
   end

   generate
   if (NUM_SLAVE == 4) begin
     always @(posedge clk)
     begin
       avmm_mst_readdata <=({32{slave_sel[3]}} & avmm_slv_readdata[(32*3)+31:(32*3)+0]) |
                           ({32{slave_sel[2]}} & avmm_slv_readdata[(32*2)+31:(32*2)+0]) |
                           ({32{slave_sel[1]}} & avmm_slv_readdata[(32*1)+31:(32*1)+0]) |
                           ({32{slave_sel[0]}} & avmm_slv_readdata[(32*0)+31:(32*0)+0]) ;
     end
   end else begin
     always @(posedge clk)
     begin
       avmm_mst_readdata <=({32{slave_sel[5]}} & avmm_slv_readdata[(32*5)+31:(32*5)+0]) |
                           ({32{slave_sel[4]}} & avmm_slv_readdata[(32*4)+31:(32*4)+0]) |
                           ({32{slave_sel[3]}} & avmm_slv_readdata[(32*3)+31:(32*3)+0]) |
                           ({32{slave_sel[2]}} & avmm_slv_readdata[(32*2)+31:(32*2)+0]) |
                           ({32{slave_sel[1]}} & avmm_slv_readdata[(32*1)+31:(32*1)+0]) |
                           ({32{slave_sel[0]}} & avmm_slv_readdata[(32*0)+31:(32*0)+0]) ;
     end
   end
   endgenerate

`endif
endmodule //DR_F_AVMM_INTERCONNECT

`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "79ztKTnV2ny/eHpgZYepe9G9wwwdeg9hTOdIyLdh4JTuZZ1DT50cJOPNpZ0w+c5aVHkra3Tx9+PMlDB4omAPannbtZoJG9ZekMCcgtPHzQ4GWiJfmhYoHZqap4yb9/uKlHSl9TpAcV1FKEwkrctIKziGBKSQ0S70p/jW9wD4CtM3X7P+ggV0x7Prg108lDSsdB1kGvgcQN/5kLCIPO6i12claBXdWuhbdU7Bm3JHKTol1lwsclaPxDRkOAM5iYAjg6W6uvMsX/7sV89Cd6wAUT7qBE4o+J1mCUZsoD4BUbbBEynAhYWDg6c07nhP+Hu297ax4MovoA7+yMzcn+kCPUheybidR7ly4Rg4M6FDN/XFWnA8BZBtdh15eEfzaOsOzWvLC3HQTu34+c3wTcdazRUh4ZYINXpPV2fgK5/8qRDtgEYuCRpxDi1CxAHrrX5OPGuOHQgynmoie5U/7rxL4mqb1QsIOsHn1uHCvFUU2LXwwjbQModIGzY7Dfy0FuJXhEh6JNL9eu6cawKv1aPLLH6UcXy/j/RW5ySFJ90IG633Y5YOsq9FAvtEx86Qs62E8Daz8c0CnUz02vbblsXepUaAWOeDayitz52nNdO5rwLaIe2VCfteBPtHwjde/jFk4k0hXjZdQbIY5Xwy/4ITm1ZAYtUnxR9ayoGMo8Ga6CJGLFgm5wFIHgY/ZNBumPzm6Kle1iG2YA9IPGgY8hNY7/QuH+bmhs+J/8osex7p6NLNwHUZ1FrFHeQm3hOpbPfuv/JD4KiRycYnTusbY3kYJqBwWIvtfWyrgKR9MWzzosCY+6tGAlKB7lXJHtxAjvlZLsQ2huiZO1HVT9IArvuU8UEc3FJRjfet+CQrVUguH/I4WbEPHydko5bbL6ScptQrMy0gyTyLEpzVISSM6iy2IAcUPWFNWALkrrXenJSFl/jKZUzr0VUMl6sBCzcl9LdCv7pW41j0SeeAYqvItPCxLmJUDXtEKeGj1pbgivD/qsol+iTtDGES8pmSNuWadEYj"
`endif