// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


//----------------------------------------------------------------------------- 
//  Project Name:  GDR Dynamic Reconfiguration (F-tile)                                  
//  Module Name :  dr_f_avmm_interconnect                                  
//  Author ID   :  hafeezmu                                   
//  Date        :  Wed Feb 19 10:27:00 2020   
//  Version     :  v1.0 (Initial Draft)                               
//  Description :  GDR DR AVMM interconnect is to split single AVMM to multiple    
//                 AVMM slave interfaces. This component is part of DR SIP.This  
//                 commponent use to access local AVMM1 24 channels, Global AVMM
//                 and DR CSR address space                                             
//-----------------------------------------------------------------------------  

`timescale 1 ps / 1 ps

module dr_f_scalable_decoder #(
        parameter SLAVE_NUM          = 3,
        parameter ADDR_WIDTH         = 32,
        parameter DATA_WIDTH         = 32,
        parameter BYTE_EN_WIDTH      = 4
    )(
        input   logic                                 clk,
        input   logic                                 rst_n,

        //---avmm master from Jtag---
        input   logic   [ADDR_WIDTH-1:0]              master_addr,
        input   logic                                 master_read,
        input   logic                                 master_write,
        input   logic   [DATA_WIDTH-1:0]              master_writedata,
        input   logic   [BYTE_EN_WIDTH-1:0]           master_byteenable,
        output  logic   [DATA_WIDTH-1:0]              master_readdata,
        output  logic                                 master_readdatavalid,
        output  logic                                 master_waitrequest,

        //---avmm slave IF---
        output logic   [ADDR_WIDTH-1:0]               slave_addr,
        output logic   [DATA_WIDTH-1:0]               slave_writedata,
        output logic   [BYTE_EN_WIDTH-1:0]            slave_byteenable,
        output logic   [SLAVE_NUM-1:0]                slave_read,
        output logic   [SLAVE_NUM-1:0]                slave_write,
        input  logic   [SLAVE_NUM*DATA_WIDTH-1:0]     slave_readdata,
        input  logic   [SLAVE_NUM-1:0]                slave_readdatavalid,
        input  logic   [SLAVE_NUM-1:0]                slave_waitrequest,

        //---Ctrl IF---
        input   logic   [SLAVE_NUM*ADDR_WIDTH-1:0]    slave_start_addr,
        input   logic   [SLAVE_NUM*ADDR_WIDTH-1:0]    slave_end_addr

);

//--------------------------------------------
integer i, j;
logic   [SLAVE_NUM:0]     sel, rd_vld, wr_vld, rw_done_all;
logic                     rd_req, wr_req, rd_done, wr_done, rw_done, rdata_vld_t, rdata_vld, rdata_vld_r;

logic   [ADDR_WIDTH-1:0]     master_addr_r;
logic                        master_read_r;
logic                        master_write_r;
logic   [DATA_WIDTH-1:0]     master_writedata_r;
logic   [BYTE_EN_WIDTH-1:0]  master_byteenable_r;

//--------------------------------------------
logic [1:0]     avmm_st, avmm_nst;
localparam      AVMM_IDLE        = 2'h0,
                AVMM_RW          = 2'h1,
                AVMM_RW_DONE     = 2'h2,
                AVMM_RDATA       = 2'h3;

//--------------------------------------------
always @* begin
    avmm_nst = avmm_st;
    case (avmm_st)
        AVMM_IDLE:      if (master_read | master_write)    avmm_nst = AVMM_RW;
        AVMM_RW:        if (rw_done)                       avmm_nst = AVMM_RW_DONE;
        AVMM_RW_DONE:   if (master_read_r & !rdata_vld)    avmm_nst = AVMM_RDATA;
                        else                               avmm_nst = AVMM_IDLE;
        AVMM_RDATA:     if (master_read_r & rdata_vld)     avmm_nst = AVMM_IDLE;
        default:                                           avmm_nst = AVMM_IDLE;
    endcase
end

always @ (posedge clk) begin
  if (!rst_n)     avmm_st <= AVMM_IDLE;
  else            avmm_st <= avmm_nst;
end

always @ (posedge clk) begin
  if (!rst_n)     master_readdatavalid <= 1'b0;
  else            master_readdatavalid <= rdata_vld_t & (|sel[SLAVE_NUM-1:0]);
end

always @ (posedge clk) begin
  if (!rst_n)     master_waitrequest <= 1'b1;
  else            master_waitrequest <= !((avmm_nst==AVMM_RW_DONE) | (avmm_st==AVMM_RW_DONE));
end


assign rd_req = master_read_r  & (avmm_st==AVMM_RW);
assign wr_req = master_write_r & (avmm_st==AVMM_RW);
assign slave_read  = {SLAVE_NUM{rd_req}} & sel[SLAVE_NUM-1:0];
assign slave_write = {SLAVE_NUM{wr_req}} & sel[SLAVE_NUM-1:0];

//--------------------------------------------
logic [(SLAVE_NUM+1)*DATA_WIDTH-1:0]  slave_readdata_all;
assign slave_readdata_all = (DATA_WIDTH == 8) ? {8'hde, slave_readdata} : {32'hdeadc0de, slave_readdata};

logic [DATA_WIDTH-1:0] rd_data[0:SLAVE_NUM];
always @* begin
    for (i=0; i<=SLAVE_NUM; i=i+1) begin
        for (j=0; j<DATA_WIDTH; j=j+1) begin
            rd_data[i][j] = slave_readdata_all[i*DATA_WIDTH+j];
        end
    end
end

always @ (posedge clk) begin
    for (i=0; i<=SLAVE_NUM; i=i+1) begin
        if (sel[i] & rdata_vld_t) begin
            master_readdata <= rd_data[i];
        end
    end
end

//--------------------------------------------
genvar k;
generate
    for (k=0; k<SLAVE_NUM; k=k+1) begin:SLAVE_SEL
        always @ (posedge clk) begin
            if (!rst_n) 
                sel[k] <= 1'b0;
            else if ((avmm_st==AVMM_IDLE) & (avmm_nst!=AVMM_IDLE))
                sel[k] <= (master_addr >= slave_start_addr[(k+1)*ADDR_WIDTH-1:k*ADDR_WIDTH])
                        & (master_addr <= slave_end_addr[(k+1)*ADDR_WIDTH-1:k*ADDR_WIDTH]);
        end
    end
endgenerate

assign sel[SLAVE_NUM] = ~|sel[SLAVE_NUM-1:0];
//assign wr_vld = {sel[SLAVE_NUM], ~slave_waitrequest};
assign rw_done_all = {sel[SLAVE_NUM], ~slave_waitrequest};
assign rw_done = |(rw_done_all & sel);

assign rd_vld = {sel[SLAVE_NUM], slave_readdatavalid};
assign rdata_vld_t = |(rd_vld & sel);
assign rdata_vld = rdata_vld_t | rdata_vld_r;
always @(posedge clk) begin
	if (avmm_nst==AVMM_IDLE)	rdata_vld_r <= 1'b0;
	else if (rdata_vld_t)		rdata_vld_r <= 1'b1;
end
//assign rd_done = |(rd_vld & sel);
//assign wr_done = |(wr_vld & sel);

//---------------------------------------------------------------
always @(posedge clk) begin
    if ((avmm_st==AVMM_IDLE) & (avmm_nst!=AVMM_IDLE)) begin
            master_addr_r         <= master_addr;
            master_read_r         <= master_read;
            master_write_r        <= master_write;
            master_writedata_r    <= master_writedata;
            master_byteenable_r   <= master_byteenable;
    end
end

assign slave_addr       = master_addr_r;
assign slave_writedata  = master_writedata_r;
assign slave_byteenable = master_byteenable_r;

//---------------------------------------------------------------
endmodule

`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "wuj6SUN+CWUHl7X/28Op4/b/clpjqbLY1GA2Z3hIF6AQ/N7zu6Fwoye5cSVZtCMjzq7zVIP70Vz25qwBBwoIUIbppOYuSWAZCOtwfVezNqtYswQ62n/3Hi6fWGVfeSbJS3aJs8eFyk/7NahU2OTeak7iTUPHQndN4oB5O1gSp87DXZC4zl72BeNdhBCD27/XMw5oUpTkQ1qs22Ma/P0Qj3KuL2yUnt0fdnpvUGYwREwpnPjlynw8R7Y/LQxnkmbt99EhWsYoh8RC4yaznksGsHkO6ylDabrC9dpsXVOO/qHqgskUJ8c34rnIELLw4bxcO0/J/SK0RLVqZx55uuEfHfj3PZHwJjzK3cZmJnNpiqjqMyhUokBZX/b4S+MsADq+XeNu1KzUymg/fAvquV7JtijI8dnHUHlbdE45DQSw5e8Pqz6Y5FVIUEpvnMH9y9gQkrSNYUr8n7P410xEMlF6GJLX6XZ6AfNK3yqFnHlWtxa+J8V8lkPC5OdSWBYMZSBluJYIoW1z2s9iTbpF6l9wvzZpndkbNvmcCbOLqQDsRP3VrgTh1N6stJbULLcIk5eoSZBxCXLoTA1kjpMSUKOlTJxyUTH1KOTyyCKrnwcr+fL/pAh9x0phkbNjUTwOXem0JkTPttx0HXy5EINN8PZsOniU/MtdZOOHVMzLoOiBdptMmL369WfdLgw90+NrxCejuvfG8v/GWMlyU+bxu8CK9x6HPkMNTfRJdl3jIPpozdY77hcxmFHaIN6CIpN3MMR6vTzCMlwyO+cX17hKIy+V4ZASIrrthV4Zc8osjT5gZJLu6cpVncX2mXlzfy4RQjXd4geCtdnMYT73TEWtKwtVyJPIxF4SDaMrf3w8Z1Gq+kQNm0+xhlSJ6iK3sNmJ/0Ib+4AgFd5zzg+AzmQm90cILv7NTcMUVnqpD563MHyG9nxrwSI0fvF4xZfhf8RpA4q8wKpY3t8Vvqay5xVT9JccmJ5YWsehWvVQc0czaQFhVIkUiZ9jY4ad4MXpVqC7yWS0"
`endif