// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


//----------------------------------------------------------------------------- 
//  Project Name:  GDR Dynamic Reconfiguration (F-tile)                                  
//  Module Name :  dr_f_sip                                  
//  Author ID   :  hafeezmu                                   
//  Date        :  Fri Feb 21 16:07:44 2020   
//  Version     :  v1.0 (Initial Draft)                               
//  Description :  GDR DR Soft wrapper for F-tile. This component is top block 
//                 of DR SIP that integrate all sub components except DR MUX and
//                 NIOS based DR controller subsystem.                                             
//-----------------------------------------------------------------------------
// synthesis translate_off  
`timescale 1 ps / 1 fs
// synthesis translate_on
`define SRC_MUX_AND_DR_IP_MARKUP_INTFC (* intfc_name="src_mux_and_dr_ip",   intfc_type_key="p2p",  intfc_qualifier="tile_dr" *)

(* tile_ip_sip *)
module dr_f_sip #(
    parameter DWIDTH        = 32,
    parameter AWIDTH        = 26
) (
    input   logic                   csr_clk,
    input   logic                   rst_n,

    //User Interface status
    output  logic [14:0]            o_dr_curr_profile_id,
    output  logic                   o_dr_new_cfg_applied, 
    input   logic                   i_dr_new_cfg_applied_ack, 
    output  logic                   o_dr_in_progress,
    output  logic                   o_dr_error_status,
    
    // DR Controller (NIOS) interface control/status
    output  logic                   o_dr_trigger_intr,
    input   logic                   i_dr_trigger_intr_ack,
    input   logic                   i_dr_cpu_new_cfg_applied,
    output  logic                   o_dr_cpu_new_cfg_applied_ack,
   
    // DR AIB MUX selection for profile
    (* intfc_name="user_dr_one_hot_sel", intfc_type_key="slot", intfc_subtype_key="hub" *) output  logic  [767:0]        o_dr_profile_1hot_sel,

    // SRC Reconfiguration Control Interface
    output  logic  [767:0]          o_dr_ip_1hot_selector, 
    output  logic  [31:0]           dr_tx_soft_mux_sel,
    output  logic  [31:0]           dr_tx_lane_desired_state,       
    input   wire   [31:0][2:0]      dr_tx_lane_current_state,       
    input   wire   [31:0]           dr_tx_alarm,
    output  logic  [31:0]           dr_tx_clear_alarm,
    output  logic  [31:0]           dr_rx_soft_mux_sel,
    output  logic  [31:0]           dr_rx_lane_desired_state,    
    input   wire   [31:0][2:0]      dr_rx_lane_current_state,
    input   wire   [31:0]           dr_rx_alarm,
    output  logic  [31:0]           dr_rx_clear_alarm,
    input   wire   [31:0]           dr_sip_am_gen_start,
    output  logic  [31:0]           dr_sip_am_gen_2x_ack,
    input   wire   [31:0]           dr_sip_pcs_rx_sf,
    input   wire   [31:0]           dr_sip_rx_pcs_fully_aligned,
    output  logic  [31:0]           dr_sip_ehip_signal_ok,
    output  logic  [31:0]           dr_sip_rx_ignore_lock2data,
    output  logic  [31:0]           dr_sip_freeze_tx_src,
    output  logic  [31:0]           dr_sip_freeze_rx_src,
    input   wire   [31:0]           dr_sip_freeze_tx_src_ack,
    input   wire   [31:0]           dr_sip_freeze_rx_src_ack,
    output  logic  [31:0]           dr_i_dphy_iflux_ingress_input,
    input   wire   [31:0]           dr_sip_lavmm1_block_req,
    output  logic  [31:0]           dr_sip_lavmm1_block_ack,
    
    // SRC Data Memory Reconfiguration Interface 
`ifdef DR_SIP_WITH_DR_SRC_MUX_MANUAL_INTC
    input   logic                   src_clk,
    output  logic                   pause_request, 
    input   logic                   pause_grant,
    output  logic  [15:0]           group_ram_addr,
    output  logic                   group_ram_we,
    output  logic  [31:0]           group_ram_wdata,
    input   logic  [31:0]           group_ram_rdata,
`else
`define SRC_AND_DR_IP_INTFC_MARKUP \
    (* intfc_name="src_dr_if", intfc_type_key="star_connection", intfc_subtype_key="in_secondary_top_module", wire_name_fmt="\${tile_id}_src_dr_if_\${port_name}" *)
    `SRC_AND_DR_IP_INTFC_MARKUP
    input   wire                    src_clk,
    `SRC_AND_DR_IP_INTFC_MARKUP
    output  logic                   pause_request,
    `SRC_AND_DR_IP_INTFC_MARKUP
    input   wire                    pause_grant,
    `SRC_AND_DR_IP_INTFC_MARKUP
    output  logic  [15:0]           group_ram_addr,
    `SRC_AND_DR_IP_INTFC_MARKUP
    output  logic                   group_ram_we,
    `SRC_AND_DR_IP_INTFC_MARKUP
    output  logic  [31:0]           group_ram_wdata,
    `SRC_AND_DR_IP_INTFC_MARKUP
    input   wire   [31:0]           group_ram_rdata,
`undef SRC_AND_DR_IP_INTFC_MARKUP
`endif

    // Master 0 (DR Controller - NIOS subsystem interface)
    input   logic   [DWIDTH-1:0]    i_dr_cpu_avmm_writedata,
    input   logic                   i_dr_cpu_avmm_read,
    input   logic   [AWIDTH-1:0]    i_dr_cpu_avmm_address,
    input   logic                   i_dr_cpu_avmm_write,
    output  logic   [DWIDTH-1:0]    o_dr_cpu_avmm_readdata,
    output  logic                   o_dr_cpu_avmm_waitrequest,
    output  logic                   o_dr_cpu_avmm_readdata_valid,

    // Master 1 (Host software interface)
    input   logic   [DWIDTH-1:0]    i_dr_host_avmm_writedata,
    input   logic                   i_dr_host_avmm_read,
    input   logic   [AWIDTH-1:0]    i_dr_host_avmm_address,
    input   logic                   i_dr_host_avmm_write,
    output  logic   [DWIDTH-1:0]    o_dr_host_avmm_readdata,
    output  logic                   o_dr_host_avmm_waitrequest,
    output  logic                   o_dr_host_avmm_readdata_valid,

    // AVMM2/Global AVMM ports to/from hip
    output  logic                   hip_avmm_read_real,
    input   logic   [7:0]           hip_avmm_readdata_real,
    input   logic                   hip_avmm_readdatavalid_real,
    output  logic   [20:0]          hip_avmm_reg_addr_real,
    input   logic   [4:0]           hip_avmm_reserved_out_real,
    output  logic                   hip_avmm_write_real,
    output  logic   [7:0]           hip_avmm_writedata_real,
    input   logic                   hip_avmm_writedone_real,
    input   logic                   pld_avmm2_busy_real,
    output  logic                   pld_avmm2_clk_rowclk_real,
    input   logic                   pld_avmm2_cmdfifo_wr_full_real,
    input   logic                   pld_avmm2_cmdfifo_wr_pfull_real,
    output  logic                   pld_avmm2_request_real,
    input   logic                   pld_pll_cal_done_real,
    // below are unused ports in hip mode
    output  logic                   pld_avmm2_write_real,
    output  logic                   pld_avmm2_read_real,
    output  logic   [8:0]           pld_avmm2_reg_addr_real,
    input   logic   [7:0]           pld_avmm2_readdata_real,
    output  logic   [7:0]           pld_avmm2_writedata_real,
    input   logic                   pld_avmm2_readdatavalid_real,
    output  logic   [5:0]           pld_avmm2_reserved_in_real,
    input   logic                   pld_avmm2_reserved_out_real,
    
    //  for AVMM1 bb ports
    input   logic   [23:0]          pld_avmm1_busy,
    output  logic   [23:0]          pld_avmm1_clk_rowclk,
    input   logic   [23:0]          pld_avmm1_cmdfifo_wr_full,
    input   logic   [23:0]          pld_avmm1_cmdfifo_wr_pfull,
    output  logic   [23:0]          pld_avmm1_read,
    input   logic   [(24*8)-1:0]    pld_avmm1_readdata,
    input   logic   [23:0]          pld_avmm1_readdatavalid,
    output  logic   [(24*10)-1:0]   pld_avmm1_reg_addr,
    output  logic   [23:0]          pld_avmm1_request,
    output  logic   [(24*9)-1:0]    pld_avmm1_reserved_in,
    input   logic   [(24*3)-1:0]    pld_avmm1_reserved_out,
    output  logic   [23:0]          pld_avmm1_write,
    output  logic   [(24*8)-1:0]    pld_avmm1_writedata,
    input   logic   [23:0]          pld_chnl_cal_done,
    input   logic   [23:0]          pld_hssi_osc_transfer_en
);

//-----------------------------------------------------------------------------
//Declarations
//-----------------------------------------------------------------------------

    // For acks as per SRC HAS (Fully in reset)
    logic   [31:0]          i_dr_tx_rst_ack;
    logic   [31:0]          i_dr_rx_rst_ack;
    // For operational state (Full out of reset)
    logic   [31:0]          i_dr_tx_operational;
    logic   [31:0]          i_dr_rx_operational;
    // For system clock stable (Reset control initialized)
    logic   [31:0]          i_dr_tx_reset_init;
    logic   [31:0]          i_dr_rx_reset_init;

    logic                   rst_n_sync;
    logic                   rst_n_src_clk_sync;
    logic                   rst_n_sync_sync;
    
    logic   [DWIDTH-1:0]    avmm_mst_writedata;
    logic                   avmm_mst_read;
    logic   [AWIDTH-1:0]    avmm_mst_address;
    logic                   avmm_mst_write;
    logic   [DWIDTH-1:0]    avmm_mst_readdata;
    logic                   avmm_mst_waitrequest;
    logic                   avmm_mst_readdata_valid;

    // DR CSR Slave Interface AVMM Ports
    logic  [AWIDTH-1:0]     dr_csr_avmm_addr;
    logic                   dr_csr_avmm_write;
    logic                   dr_csr_avmm_read;
    logic  [31:0]           dr_csr_avmm_wdata;
    logic  [3:0]            dr_csr_avmm_byte_en;
    logic  [31:0]           dr_csr_avmm_rdata;
    logic                   dr_csr_avmm_rdatavld;
    logic                   dr_csr_avmm_waitrequest;

    // AVMM 32to8 IF Master 32bit Interface AVMM Ports
    logic   [24:0]          global_avmm32b_addr;
    logic                   global_avmm32b_write;
    logic                   global_avmm32b_read;
    logic   [31:0]          global_avmm32b_wdata;
    logic   [3:0]           global_avmm32b_byte_en;
    logic   [31:0]          global_avmm32b_rdata;
    logic                   global_avmm32b_rdatavld;
    logic                   global_avmm32b_waitreq;

    // GLOBAL_AVMM_SVIF Slave Interface AVMM Ports
    logic   [20:0]          global_avmm8b_addr;
    logic                   global_avmm8b_write;
    logic                   global_avmm8b_read;
    logic   [7:0]           global_avmm8b_wdata;
    logic   [7:0]           global_avmm8b_rdata;
    logic                   global_avmm8b_rdatavld;
    logic                   global_avmm8b_waitreq;

    // MAIB_AVMM1 32to8 IF Master 32bit Interface AVMM Ports
    logic   [22:0]          maib_avmm32b_addr;
    logic                   maib_avmm32b_write;
    logic                   maib_avmm32b_read;
    logic   [31:0]          maib_avmm32b_wdata;
    logic   [3:0]           maib_avmm32b_byte_en;
    logic   [31:0]          maib_avmm32b_rdata;
    logic                   maib_avmm32b_rdatavld;
    logic                   maib_avmm32b_waitreq;

    // MAIB_AVMM1_SVIF Slave Interface AVMM Ports
    logic                   maib_avmm8b_slave_enable;
    logic   [22:0]          maib_avmm8b_addr;
    logic                   maib_avmm8b_write;
    logic                   maib_avmm8b_read;
    logic   [7:0]           maib_avmm8b_wdata;
    logic   [7:0]           maib_avmm8b_rdata;
    logic                   maib_avmm8b_rdatavld;
    logic                   maib_avmm8b_waitreq;
    logic   [8:0]           maib_avmm32to8_slv_addr_t;
    
    logic   [22:0]          avmm_slv_maib_address;
    logic   [23:0]          avmm_slv_maib_read;
    logic   [23:0]          avmm_slv_maib_write;
    logic   [7:0]           avmm_slv_maib_writedata;
    logic   [(24*8)-1:0]    avmm_slv_maib_readdata;
    logic   [23:0]          avmm_slv_maib_waitrequest;
    logic   [23:0]          avmm_slv_maib_readdata_valid;
    
    logic   [(24*23)-1:0]   maib_start_addr;
    logic   [(24*23)-1:0]   maib_end_addr;
       
    // DR MUX selection for profile
    logic  [23:0][31:0]     dr_profile_1hot_sel;
    logic  [10:0]           avmm32to8_slv_addr_t;

    // SRC AVMM Reconfiguration Interface 
    logic  [31:0]           dr_src_avmm_writedata;
    logic                   dr_src_avmm_read;
    logic  [15:0]           dr_src_avmm_address;
    logic                   dr_src_avmm_write;
    logic  [31:0]           dr_src_avmm_readdata;
    logic                   dr_src_avmm_waitrequest;
    logic                   dr_src_avmm_readdata_valid;
    logic  [31:0]           dr_avmm_time_out;
    logic                   dr_avmm1_maib_error;
    logic                   dr_gavmm_error;
    
    logic  [31:0][2:0]      dr_tx_lane_current_state_sync;
    logic  [31:0][2:0]      dr_rx_lane_current_state_sync;    
    logic  [31:0]           dr_tx_alarm_sync;  
    logic  [31:0]           dr_rx_alarm_sync;
    logic  [31:0]           dr_sip_am_gen_start_sync;
    logic  [31:0]           dr_sip_pcs_rx_sf_sync;
    logic  [31:0]           dr_sip_rx_pcs_fully_aligned_sync;
    logic  [31:0]           dr_sip_freeze_tx_src_ack_sync;
    logic  [31:0]           dr_sip_freeze_rx_src_ack_sync; 
    logic                   pause_grant_sync;
  
    logic  [767:0]          o_dr_profile_1hot_sel_int;
    logic  [31:0]           dr_tx_soft_mux_sel_int;
    logic  [31:0]           dr_tx_lane_desired_state_int;       
    logic  [31:0]           dr_tx_clear_alarm_int;
    logic  [31:0]           dr_rx_soft_mux_sel_int;
    logic  [31:0]           dr_rx_lane_desired_state_int;    
    logic  [31:0]           dr_rx_clear_alarm_int;
    
    // Temporary assignment 
    // SRC Reconfiguration Control Interface     
    assign  dr_sip_am_gen_2x_ack = {32{1'b0}};
    assign  dr_sip_ehip_signal_ok = {32{1'b0}}; //i_dr_rx_operational & ~dr_sip_pcs_rx_sf_sync;
    assign  dr_sip_rx_ignore_lock2data = {32{1'b0}};
    assign  dr_sip_freeze_tx_src = {32{1'b0}};
    assign  dr_sip_freeze_rx_src = {32{1'b0}};
    assign  dr_i_dphy_iflux_ingress_input = {32{1'b0}};
    assign  dr_sip_lavmm1_block_ack = {32{1'b0}};
    
//-----------------------------------------------------------------------------
// Syncronization of Async reset with DR CSR clock and SRC clock
//-----------------------------------------------------------------------------
    dr_f_reset_synchronizer  dr_rst_n_sync (
        .aclr_n                 (rst_n),
        .clk                    (csr_clk),
        .aclr_sync_n            (rst_n_sync)
    );

    dr_f_xcvr_resync_std #(
        .SYNC_CHAIN_LENGTH  (3),
        .WIDTH              (1),
        .INIT_VALUE         (0)
    ) dr_rst_src_clk_sync (
        .clk                (src_clk),
        .reset              (1'b0),
        .d                  (rst_n_sync),
        .q                  (rst_n_src_clk_sync)
    );
    
    dr_f_xcvr_resync_std #(
        .SYNC_CHAIN_LENGTH  (3),
        .WIDTH              (1),
        .INIT_VALUE         (0)
    ) dr_rst_n_sync_sync (
        .clk                (csr_clk),
        .reset              (1'b0),
        .d                  (rst_n_src_clk_sync),
        .q                  (rst_n_sync_sync)
    );
        
   
//-----------------------------------------------------------------------------
// Instantiation of DR Controller and Host Controller Arbiteration block
//-----------------------------------------------------------------------------

    dr_f_avmm_arbiter_2to1 #(
       .DWIDTH  (DWIDTH),
       .AWIDTH  (AWIDTH)
    ) dr_ctrl_host_avmm_arbiter_inst (
       .csr_clk                          (csr_clk),
       .rst_n                            (rst_n_sync),
       .timeout_en                       (1'b1),
       .avmm_timeout                     (dr_avmm_time_out),

       // Master 0 (DR CSR Interface)
       .m0_read                          (dr_csr_avmm_read),
       .m0_write                         (dr_csr_avmm_write),
       .m0_address                       (dr_csr_avmm_addr),
       .m0_write_data                    (dr_csr_avmm_wdata),
       .m0_read_data                     (dr_csr_avmm_rdata),
       .m0_read_valid                    (dr_csr_avmm_rdatavld),
       .m0_wait_request                  (dr_csr_avmm_waitrequest),
       .m0_timeout                       (),

       // Master 1
       .m1_read                          (i_dr_host_avmm_read),
       .m1_write                         (i_dr_host_avmm_write),
       .m1_address                       (i_dr_host_avmm_address),
       .m1_write_data                    (i_dr_host_avmm_writedata),
       .m1_read_data                     (o_dr_host_avmm_readdata),
       .m1_read_valid                    (o_dr_host_avmm_readdata_valid),
       .m1_wait_request                  (o_dr_host_avmm_waitrequest),
       .m1_timeout                       (),

       // Slave
       .s_read                           (avmm_mst_read),
       .s_write                          (avmm_mst_write),
       .s_address                        (avmm_mst_address),
       .s_write_data                     (avmm_mst_writedata),
       .s_read_data                      (avmm_mst_readdata),
       .s_read_valid                     (avmm_mst_readdata_valid),
       .s_wait_request                   (avmm_mst_waitrequest)
    ); //dr_ctrl_host_avmm_arbiter_inst


//-----------------------------------------------------------------------------
// Instantiation of Address Decoder block between various AVMM Slaves
//-----------------------------------------------------------------------------

    dr_f_avmm_decoder #(
       .SLAVE_NUM(4)
    ) dr_avmm_decoder_inst (
       .csr_clk                          (csr_clk),
       .rst_n                            (rst_n_sync),
       .avmm_timeout                     (dr_avmm_time_out),
      
       // Master 0
       .i_avmm_mst_writedata             (i_dr_cpu_avmm_writedata),
       .i_avmm_mst_read                  (i_dr_cpu_avmm_read),
       .i_avmm_mst_address               (i_dr_cpu_avmm_address),
       .i_avmm_mst_write                 (i_dr_cpu_avmm_write),
       .o_avmm_mst_readdata              (o_dr_cpu_avmm_readdata),
       .o_avmm_mst_waitrequest           (o_dr_cpu_avmm_waitrequest),
       .o_avmm_mst_readdata_valid        (o_dr_cpu_avmm_readdata_valid),
          
       // GLOBAL_AVMM_SVIF Slave Interface AVMM Ports
       .o_dr_global_avmm_ictrl_addr      (global_avmm32b_addr),
       .o_dr_global_avmm_ictrl_write     (global_avmm32b_write),
       .o_dr_global_avmm_ictrl_read      (global_avmm32b_read),
       .o_dr_global_avmm_ictrl_wdata     (global_avmm32b_wdata),
       .o_dr_global_avmm_ictrl_byte_en   (global_avmm32b_byte_en),
       .i_dr_global_avmm_ictrl_rdata     (global_avmm32b_rdata),
       .i_dr_global_avmm_ictrl_rdatavld  (global_avmm32b_rdatavld),
       .i_dr_global_avmm_ictrl_waitreq   (global_avmm32b_waitreq),

       // MAIB_AVMM_SVIF Slave Interface AVMM1 Ports
       .o_maib_avmm32b_addr              (maib_avmm32b_addr),
       .o_maib_avmm32b_write             (maib_avmm32b_write),
       .o_maib_avmm32b_read              (maib_avmm32b_read),
       .o_maib_avmm32b_wdata             (maib_avmm32b_wdata),
       .o_maib_avmm32b_byte_en           (maib_avmm32b_byte_en),
       .i_maib_avmm32b_rdata             (maib_avmm32b_rdata),
       .i_maib_avmm32b_rdatavld          (maib_avmm32b_rdatavld),
       .i_maib_avmm32b_waitreq           (maib_avmm32b_waitreq),
    
       // DR CSR Slave Interface AVMM Ports
       .o_dr_csr_avmm_addr               (dr_csr_avmm_addr),
       .o_dr_csr_avmm_write              (dr_csr_avmm_write),
       .o_dr_csr_avmm_read               (dr_csr_avmm_read),
       .o_dr_csr_avmm_wdata              (dr_csr_avmm_wdata),
       .o_dr_csr_avmm_byte_en            (dr_csr_avmm_byte_en),
       .i_dr_csr_avmm_rdata              (dr_csr_avmm_rdata),
       .i_dr_csr_avmm_rdatavld           (dr_csr_avmm_rdatavld),
       .i_dr_csr_avmm_waitrequest        (dr_csr_avmm_waitrequest),
       
       // SRC Slave Interface AVMM Ports
       .o_dr_src_avmm_writedata          (dr_src_avmm_writedata),
       .o_dr_src_avmm_read               (dr_src_avmm_read),
       .o_dr_src_avmm_address            (dr_src_avmm_address),
       .o_dr_src_avmm_write              (dr_src_avmm_write),
       .i_dr_src_avmm_readdata           (dr_src_avmm_readdata),
       .i_dr_src_avmm_waitrequest        (dr_src_avmm_waitrequest),
       .i_dr_src_avmm_readdata_valid     (dr_src_avmm_readdata_valid)
   
    ); //dr_avmm_decoder_inst
    
//-----------------------------------------------------------------------------
// Instantiation of 32to8 and 8to32 data conversion AVMM interface block 
//-----------------------------------------------------------------------------
    dr_f_gavmm_32to8 #(
       .TIMEOUT(5000)
    ) dr_gavmm_32to8_inst (
       .clk                              (csr_clk),
       .rst_n                            (rst_n_sync),
       .avmm_timeout                     (dr_avmm_time_out),
       .avmm_error                       (dr_gavmm_error),
       
       .master_read                      (global_avmm32b_read),
       .master_write                     (global_avmm32b_write),
       .master_address                   ({7'b0,global_avmm32b_addr}), 
       .master_write_data                (global_avmm32b_wdata),
       .master_byte_enable               (global_avmm32b_byte_en),
       .master_read_valid                (global_avmm32b_rdatavld),
       .master_wait                      (global_avmm32b_waitreq),
       .master_read_data                 (global_avmm32b_rdata),

       .slave_enable                     (1'b1),
       .slave_read                       (global_avmm8b_read),
       .slave_write                      (global_avmm8b_write),
       .slave_address                    ({avmm32to8_slv_addr_t,global_avmm8b_addr}), //TODO
       .slave_write_data                 (global_avmm8b_wdata),
       .slave_read_valid                 (global_avmm8b_rdatavld),
       .slave_wait                       (global_avmm8b_waitreq),
       .slave_read_data                  (global_avmm8b_rdata)
    ); //dr_avmm_32to8_inst

    assign global_avmm8b_rdatavld = global_avmm8b_read & ~global_avmm8b_waitreq;

//-----------------------------------------------------------------------------
// Instantiate avmm2 core logic 
//-----------------------------------------------------------------------------
`ifndef DR_SIP_ONLY
    ctf_avmm2_soft_logic
    #(  .avmm_interfaces (1),  //Number of AVMM interfaces required - one for each bonded_lane, PLL, and Master CGB
        .rcfg_enable     (1)   //Enable/disable reconfig interface in the XCVR PMA or PCIe IPs
     ) dr_avmm2_cl_inst (
       // AVMM slave interface signals (user)
       .avmm_clk                        (csr_clk) ,
       .avmm_reset                      (~rst_n),
       .avmm_writedata                  (global_avmm8b_wdata), 
       .avmm_address                    (global_avmm8b_addr), 
       .avmm_write                      (global_avmm8b_write),
       .avmm_read                       (global_avmm8b_read),
       .avmm_readdata                   (global_avmm8b_rdata), 
       .avmm_waitrequest                (global_avmm8b_waitreq),
       //AVMM interface busy with calibration
       .avmm_busy                       (),
     
       // Expose clkchnl to wire up with pld_adapt avmmclk for Place and Route in Fitter
       .avmm_clkchnl                    (),

       // ports to/from hip ports of building block
       .hip_avmm_read_real              (hip_avmm_read_real),
       .hip_avmm_readdata_real          (hip_avmm_readdata_real),
       .hip_avmm_readdatavalid_real     (hip_avmm_readdatavalid_real),
       .hip_avmm_reg_addr_real          (hip_avmm_reg_addr_real),
       .hip_avmm_reserved_out_real      (hip_avmm_reserved_out_real),
       .hip_avmm_write_real             (hip_avmm_write_real),
       .hip_avmm_writedata_real         (hip_avmm_writedata_real),
       .hip_avmm_writedone_real         (hip_avmm_writedone_real),
       .pld_avmm2_busy_real             (pld_avmm2_busy_real),
       .pld_avmm2_clk_rowclk_real       (pld_avmm2_clk_rowclk_real),
       .pld_avmm2_cmdfifo_wr_full_real  (pld_avmm2_cmdfifo_wr_full_real),
       .pld_avmm2_cmdfifo_wr_pfull_real (pld_avmm2_cmdfifo_wr_pfull_real),
       .pld_avmm2_request_real          (pld_avmm2_request_real),
       .pld_pll_cal_done_real           (pld_pll_cal_done_real),
       // below are unused ports in hip mode
       .pld_avmm2_write_real            (pld_avmm2_write_real),
       .pld_avmm2_read_real             (pld_avmm2_read_real),
       .pld_avmm2_reg_addr_real         (pld_avmm2_reg_addr_real),
       .pld_avmm2_readdata_real         (pld_avmm2_readdata_real),
       .pld_avmm2_writedata_real        (pld_avmm2_writedata_real),
       .pld_avmm2_readdatavalid_real    (pld_avmm2_readdatavalid_real),
       .pld_avmm2_reserved_in_real      (pld_avmm2_reserved_in_real),
       .pld_avmm2_reserved_out_real     (pld_avmm2_reserved_out_real)
    );
`endif

//-----------------------------------------------------------------------------
// Instantiation of 32to8 and 8to32 data conversion AVMM1 MAIB interface block 
//-----------------------------------------------------------------------------  
    dr_f_avmm_32to8 #(
       .TIMEOUT(5000)
    ) dr_avmm1_maib_32to8_inst (
       .clk                              (csr_clk),
       .rst_n                            (rst_n_sync),
       .avmm_timeout                     (dr_avmm_time_out),
       .avmm_error                       (dr_avmm1_maib_error),
       
       .master_read                      (maib_avmm32b_read),
       .master_write                     (maib_avmm32b_write),
       .master_address                   ({9'h0,maib_avmm32b_addr[22:0]}), 
       .master_write_data                (maib_avmm32b_wdata),
       .master_byte_enable               (maib_avmm32b_byte_en),
       .master_read_valid                (maib_avmm32b_rdatavld),
       .master_wait                      (maib_avmm32b_waitreq),
       .master_read_data                 (maib_avmm32b_rdata),

       .slave_enable                     (maib_avmm8b_slave_enable),
       .slave_read                       (maib_avmm8b_read),
       .slave_write                      (maib_avmm8b_write),
       .slave_address                    ({maib_avmm32to8_slv_addr_t,maib_avmm8b_addr}), //TODO
       .slave_write_data                 (maib_avmm8b_wdata),
       .slave_read_valid                 (maib_avmm8b_rdatavld),
       .slave_wait                       (maib_avmm8b_waitreq),
       .slave_read_data                  (maib_avmm8b_rdata)
    ); //dr_avmm1_maib_32to8_inst

   assign maib_avmm8b_slave_enable = (maib_avmm32b_addr[17:0] >= 18'h2C0) & (maib_avmm32b_addr[17:0] <= 18'h3FC);
   
//-----------------------------------------------------------------------------
// Instantiation of AVMM1 Address Decoder block between various MAIB Slaves
//-----------------------------------------------------------------------------   
    genvar j;
    generate 
      for (j = 0; j < 24; j = j+1) begin: MAIB_BASE_ADDR
         assign maib_start_addr[(j*23)+:23] = {j, 18'h0_0000};
         assign maib_end_addr[(j*23)+:23]   = {j, 18'h3_FFFF};
      end: MAIB_BASE_ADDR
    endgenerate
    
    //decoder instance
    dr_f_scalable_decoder #(
        .SLAVE_NUM     (24),
        .ADDR_WIDTH    (23),
        .DATA_WIDTH    (8),
        .BYTE_EN_WIDTH (1)
    ) dr_f_avmm1_maib_decoder_inst (
       .clk                              (csr_clk),
       .rst_n                            (rst_n_sync),
       //master ports                    
       .master_addr                      (maib_avmm8b_addr),
       .master_read                      (maib_avmm8b_read),
       .master_write                     (maib_avmm8b_write),
       .master_writedata                 (maib_avmm8b_wdata),
       .master_byteenable                (1'b1),
       .master_readdata                  (maib_avmm8b_rdata),
       .master_readdatavalid             (maib_avmm8b_rdatavld),
       .master_waitrequest               (maib_avmm8b_waitreq),
       //slave ports                     
       .slave_addr                       (avmm_slv_maib_address),
       .slave_writedata                  (avmm_slv_maib_writedata),
       .slave_byteenable                 (),
       .slave_read                       (avmm_slv_maib_read),
       .slave_write                      (avmm_slv_maib_write),
       .slave_readdata                   (avmm_slv_maib_readdata),
       .slave_readdatavalid              (avmm_slv_maib_readdata_valid),
       .slave_waitrequest                (avmm_slv_maib_waitrequest),
       //---Ctrl IF---                   
       .slave_start_addr                 (maib_start_addr),
       .slave_end_addr                   (maib_end_addr)
    ); //dr_f_avmm1_maib_decoder_inst
  
    assign avmm_slv_maib_readdata_valid = avmm_slv_maib_read & ~avmm_slv_maib_waitrequest;
    
//-----------------------------------------------------------------------------
// Instantiate avmm1 core logic 
//-----------------------------------------------------------------------------
`ifndef DR_SIP_ONLY
    ctf_avmm1_soft_logic #(
        .avmm_interfaces (24), //Number of AVMM interfaces required - one for each bonded_lane, PLL, and Master CGB
        .rcfg_enable     (1)   //Enable/disable reconfig interface in the Native PHY or PLL IP
    )  ctf_avmm1_soft_logic_inst (
       //PORT_LIST_START
       // AVMM slave interface signals (user)
       .avmm_clk                         ({24{csr_clk}}),
       .avmm_reset                       ({24{~rst_n}}),
       .avmm_writedata                   ({24{avmm_slv_maib_writedata}}),
       .avmm_address                     ({24{2'b11,avmm_slv_maib_address[7:0]}}),  //10 bits. addr[9:8]:2'b11 for MAIB, rest for F-tile, addr[8] is Dword Access
       .avmm_reservedin                  ({24{avmm_slv_maib_address[14:8],2'b00}}), //9 bits
       .avmm_write                       (avmm_slv_maib_write),
       .avmm_read                        (avmm_slv_maib_read),
       .avmm_readdata                    (avmm_slv_maib_readdata),
       .avmm_waitrequest                 (avmm_slv_maib_waitrequest),
       // Signals from AVMM1 building block
       .pld_avmm1_busy_real              (pld_avmm1_busy),
       .pld_avmm1_cmdfifo_wr_full_real   (pld_avmm1_cmdfifo_wr_full),
       .pld_avmm1_cmdfifo_wr_pfull_real  (pld_avmm1_cmdfifo_wr_pfull),
       .pld_avmm1_readdata_real          (pld_avmm1_readdata), // 8 bits
       .pld_avmm1_readdatavalid_real     (pld_avmm1_readdatavalid),
       .pld_avmm1_reserved_out_real      (pld_avmm1_reserved_out), // 3 bits
       .pld_chnl_cal_done_real           (pld_chnl_cal_done), 
       .pld_hssi_osc_transfer_en_real    (pld_hssi_osc_transfer_en),
       // Signals to AVMM1 building block
       .pld_avmm1_clk_rowclk_real        (pld_avmm1_clk_rowclk),
       .pld_avmm1_read_real              (pld_avmm1_read),
       .pld_avmm1_reg_addr_real          (pld_avmm1_reg_addr), // 10 bits
       .pld_avmm1_request_real           (pld_avmm1_request),
       .pld_avmm1_reserved_in_real       (pld_avmm1_reserved_in), // 9 bits
       .pld_avmm1_write_real             (pld_avmm1_write),
       .pld_avmm1_writedata_real         (pld_avmm1_writedata) // 8 bits
       //PORT_LIST_END
    );
`endif
  
//-----------------------------------------------------------------------------
// Instantiation of DR Controller CSR block
//-----------------------------------------------------------------------------

    dr_f_csr  dr_csr_inst (
    
       .csr_clk                          (csr_clk),
       .rst_n                            (rst_n_sync),
      
       //Bus Interface
       .writedata                        (avmm_mst_writedata),
       .read                             (avmm_mst_read),
       .write                            (avmm_mst_write),
       .byteenable                       (4'hF),
       .readdata                         (avmm_mst_readdata),
       .readdatavalid                    (avmm_mst_readdata_valid),
       .address                          (avmm_mst_address),
       .waitrequest                      (avmm_mst_waitrequest),

       //User Interface status
       .o_dr_curr_profile_id             (o_dr_curr_profile_id),
       .o_dr_new_cfg_applied             (o_dr_new_cfg_applied), 
       .i_dr_new_cfg_applied_ack         (i_dr_new_cfg_applied_ack), 
       .o_dr_in_progress                 (o_dr_in_progress),
       
       // DR Controller (NIOS) interface control/status
       .o_dr_trigger_intr                (o_dr_trigger_intr),
       .i_dr_trigger_intr_ack            (i_dr_trigger_intr_ack),
       .i_dr_cpu_new_cfg_applied         (i_dr_cpu_new_cfg_applied),
       .o_dr_cpu_new_cfg_applied_ack     (o_dr_cpu_new_cfg_applied_ack),

       // DR HIP interface control/status
       .i_dr_tx_rst_ack                  (i_dr_tx_rst_ack),
       .i_dr_rx_rst_ack                  (i_dr_rx_rst_ack),
       .i_dr_tx_operational              (i_dr_tx_operational),
       .i_dr_rx_operational              (i_dr_rx_operational),
       .i_dr_tx_reset_init               (i_dr_tx_reset_init),
       .i_dr_rx_reset_init               (i_dr_rx_reset_init),
       .i_dr_src_tx_alarm                (dr_tx_alarm_sync),
       .o_dr_src_tx_clear_alarm          (dr_tx_clear_alarm_int),
       .i_dr_src_rx_alarm                (dr_rx_alarm_sync),
       .o_dr_src_rx_clear_alarm          (dr_rx_clear_alarm_int),
       .i_dr_src_pause_grant             (pause_grant_sync),
       .o_dr_src_pause_request           (pause_request),
       .o_dr_src_tx_lane_desired_state   (dr_tx_lane_desired_state_int),
       .o_dr_src_rx_lane_desired_state   (dr_rx_lane_desired_state_int), 
       .o_dr_src_tx_soft_mux_sel         (dr_tx_soft_mux_sel_int),
       .o_dr_src_rx_soft_mux_sel         (dr_rx_soft_mux_sel_int),

       // DR MUX selection for profile
       .o_dr_profile_1hot_sel            (dr_profile_1hot_sel),
       .o_dr_error_status                (o_dr_error_status),
       .i_dr_avmm1_maib_error            (dr_avmm1_maib_error),
       .i_dr_gavmm_error                 (dr_gavmm_error),
       .o_avmm_time_out                  (dr_avmm_time_out)
    );

//-----------------------------------------------------------------------------
// Instantiation of DR AVMM to SRC memory interface clock crossing bridge
//-----------------------------------------------------------------------------
   
    dr_f_avmm2mem_cc_bridge #(
       .DWIDTH(32),
       .AWIDTH(16)
    ) dr_f_src_avmm2mem_cc_bridge_inst (
       .i_avmm_clk                       (csr_clk),
       .i_avmm_rst_n                     (rst_n_sync_sync),
       .i_avmm_writedata                 (dr_src_avmm_writedata),
       .i_avmm_read                      (dr_src_avmm_read),
       .i_avmm_address                   (dr_src_avmm_address),
       .i_avmm_write                     (dr_src_avmm_write),
       .o_avmm_readdata                  (dr_src_avmm_readdata),
       .o_avmm_waitrequest               (dr_src_avmm_waitrequest),
       .o_avmm_readdata_valid            (dr_src_avmm_readdata_valid),
    
       .i_mem_clk                        (src_clk),
       .i_mem_rst                        (!rst_n_src_clk_sync),
       .i_mem_grant                      (pause_grant),
       .o_mem_addr                       (group_ram_addr),
       .o_mem_we                         (group_ram_we),
       .o_mem_wdata                      (group_ram_wdata),
       .i_mem_rdata                      (group_ram_rdata)
	);
        
    // Convert to single Array
    genvar i;
    generate 
      for (i = 0; i < 24; i = i+1) begin: PORT_CONVERT
         assign o_dr_profile_1hot_sel_int[(i*32)+:32] = dr_profile_1hot_sel[i];
         assign o_dr_ip_1hot_selector[(i*32)+:32] = dr_profile_1hot_sel[i];
      end: PORT_CONVERT
    endgenerate
   
    generate  
      for (i=0; i<32; i=i+1) begin: CURRENT_STATE_MAP
         // taking bit-0 for acks as per SRC HAS (Fully in reset)
         assign i_dr_tx_rst_ack[i] = dr_tx_lane_current_state_sync[i][0];
         assign i_dr_rx_rst_ack[i] = dr_rx_lane_current_state_sync[i][0];
 
         // taking bit-1 for operational state (Fully out of reset)
         assign i_dr_tx_operational[i] = dr_tx_lane_current_state_sync[i][1];
         assign i_dr_rx_operational[i] = dr_rx_lane_current_state_sync[i][1];

         // taking bit-2 for system clock stable (Reset control initialized)
         assign i_dr_tx_reset_init[i] = dr_tx_lane_current_state_sync[i][2];
         assign i_dr_rx_reset_init[i] = dr_rx_lane_current_state_sync[i][2];
      end: CURRENT_STATE_MAP
    endgenerate 
  
//-----------------------------------------------------------------------------
// Synchronize the interface status signals coming from SRC controller
//-----------------------------------------------------------------------------
    dr_f_altera_std_synchronizer_nocut #(
       .depth     (3),
       .rst_value (1)
    ) pause_grant_dr_sync  (
       .clk     (csr_clk),
       .reset_n (1'b1),
       .din     (pause_grant),
       .dout    (pause_grant_sync)
    );

`ifndef __ALTERA_STD__METASTABLE_SIM  
    dr_f_multibit_sync #(
        .WIDTH (768)
    ) dr_profile_1hot_sel_src_sync (
        .clk     (src_clk),
        .reset_n (1'b1),
        .din     (o_dr_profile_1hot_sel_int),
        .dout    (o_dr_profile_1hot_sel)
    );    
`else
    assign o_dr_profile_1hot_sel = o_dr_profile_1hot_sel_int;
`endif
    
    genvar st;
    generate
      for (st=0; st<32; st=st+1) begin: CURRENT_STATE     
        dr_f_multibit_sync #(
            .WIDTH (3)
        ) tx_lane_current_dr_sync (
            .clk     (csr_clk),
            .reset_n (1'b1),
            .din     (dr_tx_lane_current_state[st][2:0]),
            .dout    (dr_tx_lane_current_state_sync[st][2:0])
        ); 
      
        dr_f_multibit_sync #(
            .WIDTH (3)
        ) rx_lane_current_dr_sync (
            .clk     (csr_clk),
            .reset_n (1'b1),
            .din     (dr_rx_lane_current_state[st][2:0]),
            .dout    (dr_rx_lane_current_state_sync[st][2:0])
        );
      end: CURRENT_STATE
    endgenerate

    dr_f_multibit_sync #(
        .WIDTH (32)
    ) tx_alarm_dr_sync (
        .clk     (csr_clk),
        .reset_n (1'b1),
        .din     (dr_tx_alarm),
        .dout    (dr_tx_alarm_sync)
    );
       
    dr_f_multibit_sync #(
        .WIDTH (32)
    ) rx_alarm_dr_sync (
        .clk     (csr_clk),
        .reset_n (1'b1),
        .din     (dr_rx_alarm),
        .dout    (dr_rx_alarm_sync)
    );
 
    dr_f_multibit_sync #(
        .WIDTH (32)
    ) sip_am_gen_start_dr_sync (
        .clk     (csr_clk),
        .reset_n (1'b1),
        .din     (dr_sip_am_gen_start),
        .dout    (dr_sip_am_gen_start_sync)
    );
 
    dr_f_multibit_sync #(
        .WIDTH (32)
    ) sip_pcs_rx_sf_dr_sync (
        .clk     (csr_clk),
        .reset_n (1'b1),
        .din     (dr_sip_pcs_rx_sf),
        .dout    (dr_sip_pcs_rx_sf_sync)
    ); 

    dr_f_multibit_sync #(
        .WIDTH (32)
    ) sip_rx_pcs_fully_aligned_dr_sync (
        .clk     (csr_clk),
        .reset_n (1'b1),
        .din     (dr_sip_rx_pcs_fully_aligned),
        .dout    (dr_sip_rx_pcs_fully_aligned_sync)
    ); 
    
    dr_f_multibit_sync #(
        .WIDTH (32)
    ) sip_freeze_tx_src_ack_dr_sync (
        .clk     (csr_clk),
        .reset_n (1'b1),
        .din     (dr_sip_freeze_tx_src_ack),
        .dout    (dr_sip_freeze_tx_src_ack_sync)
    ); 
    
    dr_f_multibit_sync #(
        .WIDTH (32)
    ) sip_freeze_rx_src_ack_dr_sync (
        .clk     (csr_clk),
        .reset_n (1'b1),
        .din     (dr_sip_freeze_rx_src_ack),
        .dout    (dr_sip_freeze_rx_src_ack_sync)
    );
          
    dr_f_multibit_sync #(
        .WIDTH (32)
    ) dr_tx_soft_mux_sel_src_sync (
        .clk     (src_clk),
        .reset_n (1'b1),
        .din     (dr_tx_soft_mux_sel_int),
        .dout    (dr_tx_soft_mux_sel)
    );

    dr_f_multibit_sync #(
        .WIDTH (32)
    ) dr_tx_lane_desired_state_src_sync (
        .clk     (src_clk),
        .reset_n (1'b1),
        .din     (dr_tx_lane_desired_state_int),
        .dout    (dr_tx_lane_desired_state)
    );

    dr_f_multibit_sync #(
        .WIDTH (32)
    ) dr_tx_clear_alarm_src_sync (
        .clk     (src_clk),
        .reset_n (1'b1),
        .din     (dr_tx_clear_alarm_int),
        .dout    (dr_tx_clear_alarm)
    );

    dr_f_multibit_sync #(
        .WIDTH (32)
    ) dr_rx_soft_mux_sel_src_sync (
        .clk     (src_clk),
        .reset_n (1'b1),
        .din     (dr_rx_soft_mux_sel_int),
        .dout    (dr_rx_soft_mux_sel)
    );

    dr_f_multibit_sync #(
        .WIDTH (32)
    ) dr_rx_lane_desired_state_src_sync (
        .clk     (src_clk),
        .reset_n (1'b1),
        .din     (dr_rx_lane_desired_state_int),
        .dout    (dr_rx_lane_desired_state)
    );

    dr_f_multibit_sync #(
        .WIDTH (32)
    ) dr_rx_clear_alarm_src_sync (
        .clk     (src_clk),
        .reset_n (1'b1),
        .din     (dr_rx_clear_alarm_int),
        .dout    (dr_rx_clear_alarm)
    );
    
endmodule //DR_F_SIP

`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "wuj6SUN+CWUHl7X/28Op4/b/clpjqbLY1GA2Z3hIF6AQ/N7zu6Fwoye5cSVZtCMjzq7zVIP70Vz25qwBBwoIUIbppOYuSWAZCOtwfVezNqtYswQ62n/3Hi6fWGVfeSbJS3aJs8eFyk/7NahU2OTeak7iTUPHQndN4oB5O1gSp87DXZC4zl72BeNdhBCD27/XMw5oUpTkQ1qs22Ma/P0Qj3KuL2yUnt0fdnpvUGYwREw6Jhk3aWpFW7YmEMwLghxxsEcOAjoSdA+BRV5+xE1hZrE5wyum8paQMe4kCS2ldSkLGTIX+SlnNjVsxgsHKm5SIDgi+J7Jv5XbWEB51lBG9plIEpSj9XIukI+qvUoTkDtAFeMGnZNMaS+Ix+Ycy69l0XIiS93AE7vx5JHTpdBwZ7dUlCKO/TzTeA/qS3qjs52tCmsWKG26It+5jIKK40k1jDhgqTtILmlhmzMsmtNGz+sKdEJuzg/dPCoZL2ZrBvhToJ8sNIwHSiKfHIKwBIcxE8ZsxFeygAJYwJR0MA8OnKQ7Saq+KdjV9OFgv9cui3VZbIz1LolCVpw1gfJxyQfiOcKWJTTMhsplZZsME6nljW7Ph+wa1DSSYz1SUNf0e3/HY6GSCCxo945w44x7EKrxp/iJXuw4tneX5Vxdu4oCGsPyfbSVW6wHM2NvYiPPxY3rySO8POC53FkkgXXRC4h5L7TwEycs2cQPRRoRAFfOTRUW/mOiRdcCXoyE9ver8sz9w2oJu3S63P5tBk300szdYrJ5YiLSj+R4NalrLsdJ+78PBYmYSehNRzUXVae1cPogmCfX6vo+INWNLCLWScDbZayQpryG5XfcfBzM21AhPdbeKA1vgrrBIFgY6YID92oPvYSSDoVpn7TJjftRB+1pn7PloJzXDtJdQZuP4ZHy8Z/dh35CqurUfub5BRfkBaeqNF5LdHr52MBjUJuAeK1R7MtNWb+ndUauYZKeE+AA2vOuIgC98tNZPjspy/Gs+tAPD81cWY71lPMtSlaXRsP3"
`endif