// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


// Clocked priority encoder with state
//
// On each clock cycle, updates state to show which request is granted.
// Most recent grant holder is always the highest priority.
// If current grant holder is not making a request, while others are, 
// then new grant holder is always the requester with lowest bit number.
// If no requests, current grant holder retains grant state
 
// $Header$
 
`timescale 1 ns / 1 ns
//altera message_off 16753
module dphy_f_alt_xcvr_arbiter #(
	parameter width = 2
) (
	input  wire clock,
	input  wire [width-1:0] req,	// req[n] requests for this cycle
	output reg  [width-1:0] grant	// grant[n] means requester n is grantee in this cycle
);
 
	wire idle;	// idle when no requests
	wire [width-1:0] keep;	// keep[n] means requester n is requesting, and already has the grant
							// Note: current grantee is always highest priority for next grant
	wire [width-1:0] take;	// take[n] means requester n is requesting, and there are no higher-priority requests
 
	assign keep = req & grant;	// current grantee is always highest priority for next grant
	assign idle = ~| req;		// idle when no requests
 
	initial begin
		grant = 0;
	end
 
	// grant next state depends on current grant and take priority
	always @(posedge clock) begin
		grant <= 
// synthesis translate_off
                    (grant === {width{1'bx}})? {width{1'b0}} :
// synthesis translate_on
				keep				// if current grantee is requesting, gets to keep grant
				 | ({width{idle}} & grant)	// if no requests, grant state remains unchanged
				 | take;			// take applies only if current grantee is not requesting
	end
 
	// 'take' bus encodes priority.  Request with lowest bit number wins when current grantee not requesting
	assign take[0] = req[0]
					 & (~| (keep & ({width{1'b1}} << 1)));	// no 'keep' from lower-priority inputs
	genvar i;
	generate
	for (i=1; i < width; i = i + 1) begin : arb
		assign take[i] = req[i]
						 & (~| (keep & ({width{1'b1}} << (i+1))))	// no 'keep' from lower-priority inputs
						 & (~| (req & {i{1'b1}}));	// no 'req' from higher-priority inputs
	end
	endgenerate
endmodule
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "EuDK9zG4ugiX6bYeG9fkF8FUFwPRbgVIE6SygcJWoXuuaS4/+tL/ICR9sNx6uB3Kj3mUnIREp4mwUwZMubvTqsZcZoX0L/B5JV5k+lHdxdt62kmOijVOFdzJpnqmAVvPH7A96JSBQK9foFtswKYNvQUQ2+CWOyvTAMYEcYud/KOqbMyKXMgXytIkjKbvFb0Btu42lszADoIzluQfKUm8UY0h+KnrrRgSFDhz2POB1v3etwRnKVfLWX9NKNwz5cj9gluwnmMmu6nAO8x4WvWcszx4HvZO2g6p9HLAzBbmV5ruKI3jK64cNLVmgM0ZSPekcZgyBKPHMKyfYlN5dfLBIcMTMjqa7L/RoWbI7imvYtlnyKzVf4pnK4UCuneBynWaOy6IJlQ7BWyT3bSMDW2mkURJ9usyjNYzN5psTdcr7gQXjrsBm7X3JbIwx0fNsiNX4mLckK4c86RWI9c2RtKDjNErROc8Yh5FBZaKRUkfaEXImeIrW8oD3PgYe7VTABJoXBA8fsJ9UgKEsBSDKarto8/qMp10ldbxvpDHv6NBi0PJFFl6Lu8SeXMEEkUOrCOj9xkeJ6kkZIM0oCFK4zXv/JaxdP62SiSrslkRtKXirkbOW5T7I97HALnApvsjufsAUQoGDo+XwZd0gh9o1ncKkOhcAFBT+b+Ninq7BcKVYO+vPPQBD0UWUjF0XTZmTqPXI8l6h8YMlKtfsOTby614G7WQWGTuaqLPj8yQXT1p1Kn/+YQW3CmjQGrH2pPKuI1ievAoBrH3mBjnqEecr6IZOZih6gR4IwQf4fZvlATwQn/ryNWVk6kku3FnJjeyo+0EVewANa9Q4+P5OMMasmWT2/hF92j0FiiUO8QRcfeJzVNxWY5yk894IyYoj7eUTm+jwSQyWhCy6OZmI/lVVKuefMMDspISO8IOziFneCbPDt489StP67qrXTEb7dQNNmW0iZc/UE2z7w0jjEH9bw5TnBDIbO4+EPEZif4F9820rmj+3v/uZkNgDcmdAJit/Etf"
`endif