// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


module sdi_rx_dr_f #(
    parameter           VIDEO_STANDARD              = "tr",
    parameter [14:0]    MRPHY_RX_BASE_PROFILE       = 15'd1,
    parameter           RCFG_CLK_SDI_CORECLK_DIFF   = 0
) (
    input  logic        clk,
    input  logic        reset,
    input  logic        xcvr_reset_ack,
    input  logic        sdi_reconfig_request,
    input  logic [2:0]  sdi_rx_std,
    input  logic        dr_new_cfg_applied,

    input  logic        dr_avmm_readdata_valid,
    input  logic        dr_avmm_waitrequest,
    input  logic [31:0] dr_avmm_readdata,
    output logic        cdr_reconfig_busy,
    output logic        dr_new_cfg_applied_ack,
    output logic        dr_avmm_write,
    output logic        dr_avmm_read,
    output logic  [9:0] dr_avmm_address,
    output logic [31:0] dr_avmm_writedata
);

localparam [14:0] DR_12G_PROFILE  = (VIDEO_STANDARD == "mr") ? (MRPHY_RX_BASE_PROFILE+0) : 15'd1;
localparam [14:0] DR_6G_PROFILE   = (VIDEO_STANDARD == "mr") ? (MRPHY_RX_BASE_PROFILE+1) : 15'd2;
localparam [14:0] DR_3GSD_PROFILE = (VIDEO_STANDARD == "mr") ? (MRPHY_RX_BASE_PROFILE+2) : (MRPHY_RX_BASE_PROFILE+0);
localparam [14:0] DR_HD_PROFILE   = (VIDEO_STANDARD == "mr") ? (MRPHY_RX_BASE_PROFILE+3) : (MRPHY_RX_BASE_PROFILE+1);

//--------------------------------------
// state assignments
//--------------------------------------
localparam  [2:0]   IDLE                = 3'd0;
localparam  [2:0]   RD                  = 3'd1; 
localparam  [2:0]   CHK_RDDATA          = 3'd2; 
localparam  [2:0]   WR                  = 3'd3; 
localparam  [2:0]   TRANS               = 3'd4;
localparam  [2:0]   WAIT_DR_DONE        = 3'd5;
localparam [15:0]   DR_CSR_BASE_ADDR    = 16'h2000;
localparam  [1:0]   TOTAL_OFFSET        = 2'd3;

//--------------------------------------
// signals
//--------------------------------------
logic           disable_curr_profile;
logic           enable_curr_profile;
logic           trigger_reconfig;
logic           reconfig_write_done;
logic           rcfg_req_posedge_dly;
logic           reconfig_req_posedge;
logic           dr_new_cfg_applied_posedge;
logic           dr_new_cfg_applied_negedge;
logic           xcvr_reset_ack_sync;
logic           sdi_rcfg_req_sync;
logic [1:0]     num_exec;
logic [2:0]     sdi_rx_std_sync;
logic [2:0]     next_state;
logic [2:0]     state;
logic [14:0]    tmp_profile;
logic [14:0]    current_profile;

generate if (RCFG_CLK_SDI_CORECLK_DIFF)
begin : rcfg_req_sync_gen
    altera_std_synchronizer #(
        .depth      (3)
    ) rcfg_req_sync_sync_inst (
        .clk        (clk),
        .reset_n    (1'b1),
        .din        (sdi_reconfig_request),
        .dout       (sdi_rcfg_req_sync)
    );
end else begin
    assign sdi_rcfg_req_sync = sdi_reconfig_request;
end
endgenerate

altera_std_synchronizer #(
    .depth      (3)
) xcvr_reset_ack_sync_inst (
    .clk        (clk),
    .reset_n    (1'b1),
    .din        (xcvr_reset_ack),
    .dout       (xcvr_reset_ack_sync)
);

altera_std_synchronizer_bundle #(
    .width      (3),
    .depth      (3)
) u_cdr_reconfig_sel_sync (
    .clk        (clk),
    .reset_n    (1'b1),
    .din        (sdi_rx_std),
    .dout       (sdi_rx_std_sync)
);

edge_detector #(
    .EDGE_DETECT ("POSEDGE")
) reconfig_req_posedge_det_inst (
    .clk        (clk),
    .rst        (reset),
    .d          (sdi_rcfg_req_sync & xcvr_reset_ack_sync),
    .q          (reconfig_req_posedge)
);

edge_detector #(
    .EDGE_DETECT    ("POSEDGE")
) u_new_cfg_applied_posedge_det (
    .clk            (clk),
    .rst            (reset),
    .d              (dr_new_cfg_applied),
    .q              (dr_new_cfg_applied_posedge)
);

edge_detector #(
    .EDGE_DETECT    ("NEGEDGE")
) u_new_cfg_applied_negedge_det (
    .clk            (clk),
    .rst            (reset),
    .d              (dr_new_cfg_applied),
    .q              (dr_new_cfg_applied_negedge)
);

//*********************************************************************
//******Number of execution required for the whole CDR reconfig process (SDI)************
always @(posedge clk or posedge reset)
begin
    if (reset) begin
        num_exec <= TOTAL_OFFSET;
    end else begin
        if (next_state == IDLE) begin
            num_exec <= TOTAL_OFFSET;
        end else if (next_state == TRANS) begin
            num_exec <= num_exec - 1'b1;
        end
    end
end

assign disable_curr_profile = (num_exec == TOTAL_OFFSET);
assign enable_curr_profile  = (num_exec == (TOTAL_OFFSET - 1'b1));
assign trigger_reconfig     = (num_exec == 2'h1);
assign reconfig_write_done  = (num_exec == 2'h0);
//***********************************************************************************
//***************************Control State Machine***********************************
// state register
always @(posedge clk or posedge reset)
begin
    if (reset) begin
        state <= IDLE;
        rcfg_req_posedge_dly <= 1'b0;
    end else begin
        state <= next_state;
        // Delay reconfig request to state machine to allow the busy signal to arbiter 1 clock cycle earlier.
        rcfg_req_posedge_dly <= reconfig_req_posedge;
    end
end   

// next state logic
always @ (*) begin
  case(state)
    IDLE: begin
      if (rcfg_req_posedge_dly)
        next_state = RD;
      else    
        next_state = IDLE;
    end
    RD: begin
      if (dr_avmm_waitrequest)
        next_state = RD;
      else
        next_state = CHK_RDDATA;
    end
    CHK_RDDATA: begin
      if (dr_avmm_readdata_valid) begin
        if (disable_curr_profile & (dr_avmm_readdata == 32'h2))
            next_state = WR;
        else if (reconfig_write_done & (dr_avmm_readdata == 32'h0))
            next_state = WAIT_DR_DONE;
        else
            next_state = RD;
      end else begin
        next_state = CHK_RDDATA;
      end
    end
    WR: begin
      if (dr_avmm_waitrequest)
        next_state = WR;
      else
        next_state = TRANS;
    end
    TRANS : begin
       if (reconfig_write_done)
          next_state = RD;
       else
          next_state = WR;
    end
    WAIT_DR_DONE : begin
        if (dr_new_cfg_applied_negedge)
            next_state = IDLE;
        else
            next_state = WAIT_DR_DONE;
    end

    default : next_state = IDLE;
  endcase
end

//*********************************************************************
//************************** cdr_reconfig_busy****************************
always @(posedge clk or posedge reset)
begin
    if (reset)
      cdr_reconfig_busy <= 1'd0;
    else begin
      if (reconfig_req_posedge)
        cdr_reconfig_busy <= 1'b1; 
      else if (next_state == IDLE) 
        cdr_reconfig_busy <= 1'b0;
    end
end

//********************************************************************************
//*****************Generate DPRIO signals for single XCVR Interface***************
//DPRIO read
always @(posedge clk or posedge reset)
begin
    if (reset)
        dr_avmm_read  <= 1'b0; 
    else begin
        if(next_state == RD)
            dr_avmm_read  <= 1'b1; 
        else
            dr_avmm_read  <= 1'b0; 
    end
end

//DPRIO write
always @(posedge clk or posedge reset)
begin
    if (reset)
        dr_avmm_write  <= 1'b0; 
    else begin
        if(next_state == WR)
            dr_avmm_write  <= 1'b1; 
        else 
            dr_avmm_write  <= 1'b0; 
    end
end

//DPRIO writedata
always @(posedge clk or posedge reset)
begin
    if (reset) begin
        dr_avmm_writedata  <= 32'h0;
        tmp_profile <= MRPHY_RX_BASE_PROFILE;
    end else begin
        if(next_state == WR) begin
            if (disable_curr_profile)
                dr_avmm_writedata <= {17'h0, current_profile};
            else if (enable_curr_profile) begin
                if (sdi_rx_std_sync[2:1] == 2'b11) begin
                    dr_avmm_writedata <= {17'h1, DR_12G_PROFILE};
                    tmp_profile <= DR_12G_PROFILE;
                end else if (sdi_rx_std_sync[2:1] == 2'b10) begin
                    dr_avmm_writedata <= {17'h1, DR_6G_PROFILE};
                    tmp_profile <= DR_6G_PROFILE;
                end else if (sdi_rx_std_sync == 3'b001) begin
                    dr_avmm_writedata <= {17'h1, DR_HD_PROFILE};
                    tmp_profile <= DR_HD_PROFILE;
                end else begin
                    dr_avmm_writedata <= {17'h1, DR_3GSD_PROFILE};
                    tmp_profile <= DR_3GSD_PROFILE;
                end
            end else if (trigger_reconfig)
                dr_avmm_writedata <= 32'h1;
        end
    end
end

//DPRIO address
always @ (posedge clk or posedge reset)
begin
    if (reset)
        dr_avmm_address  <= 10'd0;
    else begin
        if ((disable_curr_profile && next_state == RD) || trigger_reconfig)
            dr_avmm_address <= 10'd0;
        else if (disable_curr_profile && next_state == WR)
            dr_avmm_address <= 10'd4;
        else if (enable_curr_profile)
            dr_avmm_address <= 10'd8;
    end
end

always @ (posedge clk or posedge reset)
begin
    if (reset) begin
        dr_new_cfg_applied_ack <= 1'b0;
        current_profile <= MRPHY_RX_BASE_PROFILE;
    end else begin
        dr_new_cfg_applied_ack <= dr_new_cfg_applied_posedge;
        if (dr_new_cfg_applied_posedge) begin
            current_profile <= tmp_profile;
        end
    end
end

endmodule
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "MBeNMpYGPtPl1sE+DQYudSgTsK7EAWNJwUGKRF0u7XRPhowIlls8RJfstyWmsvlOEmdSGdceoRHeaHEQLNzQQIhfBlevOTODcoCezHsI0TKITCVzhv7PoO/WuhggyXiWx6PZ9QL2w6wOssoF+/IIDkKjb9O37NLgtiIz/Y8rj7aO4WI3Z30kcN5p8CvKXADJL6/obbPPFES6e1ypucbAQpHseQQQWhKLHWIfs3d6tL6yQCXNJNMw1mfBS4rycsS6zD0RXaVTklqT6AGBPkSY61CHQgrI7GYIml6bRgWVTTjhP9BueJBE0FoMNoPC2Yi+pOl+U3KUbH/dFu6IXrfUE7e7fMWW8es7nd4tkx6xBITDuR09e0D6Pm5mJfZGfp9XuGneY501dlO9jNUY0V2PUdAvCF/yOd23vXHfluQVO1pwUYsWHlXoV0TxdYh0C3d8leIOgX3SbwoPcyOI5m2kOXWocBpmaFa/fHM9rSpfF7uDQQF2KhccMkbIh1UJhwHox8MXJLoZHt6M7yo8UT3KyqpD/Jgv3sDReZwkTArgLObhOjfdgsUJgc5xKI6qD6mYJHdtKxop9kNzMsd9lEmBGf5rG8NcXUfVUTDSgI08SWuBAkhsfy1P7ceH9dBsERRm6LH5S9S6gfvFp010vQJpJLLjJxRoPNdLr/R5z+AgVR7ZL8Xd+B8Bte1rDDcI7nu0dYBqnh0i8A3b1XcLOJ9G0I3PYzJtQV8lyuY09WVJR9Zwbhc+n8Ks6Pn//Vk5ah+hT06d5vDR3qk4EPVWkiIqR+rp/0k6/atwVUxu/hAKjcPMqRVJBnOiCDxmz1lhOcJNu406iCwuw+9MF+msHpLm+7YMgPoTKV/0G7gQfWYqBlH9L17KKDtzOgT8NaT2VCAMm2fqCnfa0gx+mUNWc3xO6aqLIvQ+EGX5NdBZI5tb5qlJPJghSqPoqaKYx9VHWYfLQHmxlWOJ+CJgVpsi83mAMiGLNmXGNqGt0A6MZatsYINldxwzD2Kyn2d1pY/Dn+pR"
`endif