// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


// The modeule converts a 32-bit data bus AVMM I/F into a native 
// GDR 8-bit data bus AVMM I/F. The msb of output addr is the indication
// of DWord_Access

module ft_avmm_32to8_bridge 
#(  
    parameter READ_PIPELINE_ENABLE = 0, 
    parameter ADDR_WIDTH = 18
) (
   // AVMM slave Port
   input  logic                  i_clk                 ,// 
   input  logic                  i_rst                 ,// 
   
   input  logic [ADDR_WIDTH-1:0] i_avmm_s32_addr           ,// 
   input  logic [31:0]           i_avmm_s32_wdata          ,//
   input  logic                  i_avmm_s32_write          ,//
   input  logic                  i_avmm_s32_read           ,//
   input  logic [3:0]            i_avmm_s32_byte_enable    ,//
   output logic [31:0]           o_avmm_s32_readdata       ,//
   output logic                  o_avmm_s32_waitrequest    ,//
   output logic                  o_avmm_s32_readdatavalid  ,// 

   // AVMMM master Port
   output  logic [ADDR_WIDTH+2:0] o_avmm_m8_addr           ,//  <1-bit DW_ACC>, <s32_addr>, <2-bit byte addr>
   output  logic [7:0]            o_avmm_m8_wdata          ,//
   output  logic                  o_avmm_m8_write          ,//
   output  logic                  o_avmm_m8_read           ,//
   input   logic [7:0]            i_avmm_m8_readdata       ,//
   input   logic                  i_avmm_m8_waitrequest     //
);

logic [1:0] byte_end, cur_byte, next_byte;  // to represent byteenables
logic       dw_acc;              // 32-bit DWord_access
logic [31:0] dw_data, next_dw_data ;

logic [2:0] cur_st, next_st;       // idle, read, write, read_done, or write_done, 

localparam ST_IDLE        = 3'b000;
localparam ST_READ        = 3'b001;
localparam ST_READ_PAUSE  = 3'b010;
localparam ST_READ_DONE   = 3'b011;
localparam ST_WRITE       = 3'b100;
localparam ST_WRITE_PAUSE = 3'b101;
localparam ST_WRITE_DONE  = 3'b110;


assign o_avmm_m8_addr  = {dw_acc, i_avmm_s32_addr[ADDR_WIDTH-1:0], cur_byte};
assign o_avmm_m8_wdata = (cur_st==ST_WRITE)? dw_data[cur_byte*8+:8] : 8'h0;
assign o_avmm_m8_write = (cur_st==ST_WRITE)? 1'b1 : 1'b0;
assign o_avmm_m8_read  = (cur_st==ST_READ)? 1'b1 : 1'b0;

assign o_avmm_s32_readdata    = (cur_st==ST_READ_DONE || (READ_PIPELINE_ENABLE && o_avmm_s32_readdatavalid))?
                                dw_data[31:0] : 32'h0;
assign o_avmm_s32_waitrequest = (cur_st==ST_READ_DONE || cur_st==ST_WRITE_DONE)? 1'b0 : 1'b1;


// convertion FSM 
//

always_ff @(posedge i_clk or posedge i_rst) begin
    if (i_rst) begin
        cur_st     <= ST_IDLE;
	dw_data    <= 32'h0;
       	cur_byte   <= 2'b00;   
        o_avmm_s32_readdatavalid <= 1'b0;
    
	dw_acc     <= 1'b0;
	byte_end   <= 2'b00;
    end else begin   // non-reset
	cur_st     <= next_st;
       	cur_byte   <= next_byte;
        dw_data    <= next_dw_data;
	o_avmm_s32_readdatavalid  <= ( READ_PIPELINE_ENABLE && (cur_st==ST_READ_DONE) )? 1'b1 : 1'b0;
	
	if (cur_st==ST_IDLE) begin
	    if (i_avmm_s32_write || i_avmm_s32_read)  begin
		dw_acc <= ( (i_avmm_s32_byte_enable==4'hf) || (i_avmm_s32_byte_enable==4'h0) )? 1'b1 : 1'b0;
		byte_end <= (i_avmm_s32_byte_enable[3])? 2'b11 : (i_avmm_s32_byte_enable[2])? 2'b10 : 
			(i_avmm_s32_byte_enable[1])? 2'b01 : (i_avmm_s32_byte_enable[0])? 2'b00 : 2'b11;
	     end  else begin	// no access
	             dw_acc       <= 1'b0;
		     byte_end     <= 2'b00;
             end 
	end       // ST_IDLE
    end           // non-reset
end     // always


always_comb  begin
    next_st      = cur_st;
    next_byte    = cur_byte;
    next_dw_data = dw_data;

    case (cur_st) 
	ST_IDLE: begin
		if (i_avmm_s32_write || i_avmm_s32_read) begin
			next_st = (i_avmm_s32_write)? ST_WRITE : ST_READ;
		        next_byte = (i_avmm_s32_byte_enable[0])? 2'b00 : (i_avmm_s32_byte_enable[1])? 2'b01 : 
			            (i_avmm_s32_byte_enable[2])? 2'b10 : (i_avmm_s32_byte_enable[3])? 2'b11 : 2'b00;
		        next_dw_data = (i_avmm_s32_write)? i_avmm_s32_wdata : 32'h0;
		    end   // read or write
	         end

	ST_READ: begin
		if ( ~i_avmm_m8_waitrequest ) begin
			next_st = (cur_byte==byte_end)? ST_READ_DONE : ST_READ_PAUSE;
			next_dw_data[cur_byte*8+:8] = i_avmm_m8_readdata;
		    end
	         end

	 ST_READ_PAUSE: begin
			next_st = ST_READ;
			next_byte = cur_byte + 1'b1;
	         end

	 ST_READ_DONE: begin
			next_st = ST_IDLE;
	         end

	 ST_WRITE: begin
		if ( ~i_avmm_m8_waitrequest ) begin
			next_st = (cur_byte==byte_end)? ST_WRITE_DONE : ST_WRITE_PAUSE;
		    end
	         end

	 ST_WRITE_PAUSE: begin
			next_st = ST_WRITE;
			next_byte = cur_byte + 1'b1;
	         end

	 ST_WRITE_DONE: begin
			next_st = ST_IDLE;
	         end

    endcase  

end    // always_comb



endmodule: ft_avmm_32to8_bridge
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "2tGVyxZMIw9h5OvjMUznwUYqTrOP39w6fMEy0tmx58wstnXD2SK+nkX9QHF0/85B2nC8OMHvnBA8JbOHsbPAhYmC1Nobd756iZcXMv6ylMYRuB0oi/dG5L7MrkxBO84IuStk5TOL/CgZbHPQrDI1ymkEO49bahXp3C3+dbWZ+6fNmo1B6zlrDwB1aY4Gzxsbu8sn4Pur4CraHkuZz1feQcJL030Bl5lAOmKhLhxSPf30pKYGnZluiJKZEGcQJbBT+xQv9SKB4cuYTKd3OyhN5g4c5BOrWYDtmY5tQv8WBMSoaS3/xmLmbg25mrqyndrzT+v8fTzWUiL2EO1j2ax697b0lt88F0OKY5Ye04L9ronzekvTNV8sVbZtWBEG9/1sAGYaRidcXNy7pi2GktHtfCmWXREnx2tOBMaZ7kUfXgZHMiLDTm48o3yMRkpZQUsSXpx0vRnGJvvT6u1diO3I5yl8uy3kXMusOPKjkLfvjYk1WabqKevG/YA8gGbIdAGE2xZqxd33NTfZ5QvRMz0LX/m/0mn/Ij8Ni0bQm5rMPC83AKDNBbsU4RXFpLlL4RsHc8e+96/7lO5WWq2GaaBGSFP484iaEmCd/AAw6XgiPR4g1rNW7TpsZ+hH3Q60k3iF9GrOQWf+OAplc2tx+QKrhjQFbNuMQRjqX8Cvj5J/Z6s6jzpJ4uzS+a+mRn09Ck4Q7itPXYR/kEUOJKK3GaBAoiISV2MhJhsUEQ1MjMg+aEMCr+6Xg7W+5glKLH+l5x6O7RS+0sUjjF2FTC+6eI0YgDugSHX9bJ2o4deHcE/ANRODhJ0RWVGuWWnk4RaVPKac/kvDwe3vyGm7TjdRxJLegUpYTIwdEdrqPLz5y6IlY/BDw04wvLeEdJFnYJzjX0NS/5X5Zja3dK1mK6z+GULLqVkHERPMJGJFOmN1WWcvmALfoZLh/YGY+OP4vjU6LaPHKRfJW6BBXKEXcBlux/JmHS8yzmJu79NPyALQbmhtmWvRRlPPQL8ftJ7scz3Nv6qX"
`endif