// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


// *********************************************************************
//
// vid_conn_dr_arbiter.sv
// 
// Description
// This module is responsible for arbitrate the AVMM interface between
// SDI Reconfig Controller and DR IP modules. 
// 
// *********************************************************************

// synthesis translate_off
`timescale 1 ps / 1 ps
// synthesis translate_on

module vid_conn_dr_arbiter #(
    parameter           NUM_DR_HOST = 1,
    parameter           PIPELINE_ENA = 0,
    parameter           AVMM_ADDRESS_WIDTH = 10, 
    parameter           AVMM_DATA_WIDTH = 32
)(
    input  logic                                        clk,
    input  logic                                        reset,
    input  logic [NUM_DR_HOST-1:0]                      reconfig_en,
                                            
    input  logic [NUM_DR_HOST-1:0]                      rcfg_mgmt_write_ch,
    input  logic [NUM_DR_HOST-1:0]                      rcfg_mgmt_read_ch,
    input  logic [NUM_DR_HOST*AVMM_ADDRESS_WIDTH-1:0]   rcfg_mgmt_address_ch,
    input  logic [NUM_DR_HOST*AVMM_DATA_WIDTH-1:0]      rcfg_mgmt_writedata_ch,
    output logic [NUM_DR_HOST*AVMM_DATA_WIDTH-1:0]      rcfg_mgmt_readdata_ch,
    output logic [NUM_DR_HOST-1:0]                      rcfg_mgmt_waitrequest_ch,
    output logic [NUM_DR_HOST-1:0]                      rcfg_mgmt_readdata_valid_ch,
    output logic [NUM_DR_HOST-1:0]                      rcfg_mgmt_new_cfg_applied_ch,
    input  logic [NUM_DR_HOST-1:0]                      rcfg_mgmt_new_cfg_applied_ack_ch,

    output logic                                        dr_avmm_write,
    output logic                                        dr_avmm_read,
    output logic [AVMM_ADDRESS_WIDTH-1:0]               dr_avmm_address,
    output logic [AVMM_DATA_WIDTH-1:0]                  dr_avmm_writedata,
    input  logic [AVMM_DATA_WIDTH-1:0]                  dr_avmm_readdata,
    input  logic                                        dr_avmm_readdata_valid,
    input  logic                                        dr_avmm_waitrequest,
    input  logic                                        dr_new_cfg_applied,
    output logic                                        dr_new_cfg_applied_ack
);

logic [5:0]  loop_counter;
logic [AVMM_ADDRESS_WIDTH-1:0] rcfg_mgmt_address_ch_array   [NUM_DR_HOST-1:0];
logic [AVMM_DATA_WIDTH-1:0]    rcfg_mgmt_writedata_ch_array [NUM_DR_HOST-1:0];
logic [AVMM_DATA_WIDTH-1:0]    rcfg_mgmt_readdata_ch_array  [NUM_DR_HOST-1:0];

// Assigning to array for easier debug
generate
for (genvar i=0; i<NUM_DR_HOST; i=i+1) begin : gen_array
    assign rcfg_mgmt_address_ch_array[i]    = rcfg_mgmt_address_ch[i*10+:10];
    assign rcfg_mgmt_writedata_ch_array[i]  = rcfg_mgmt_writedata_ch[i*32+:32];
    assign rcfg_mgmt_readdata_ch[i*32+:32]  = rcfg_mgmt_readdata_ch_array[i];
end
endgenerate

// Round robin counter to look for next reconfig host
always @ (posedge clk or posedge reset)
begin
    if (reset) begin
        loop_counter <= 6'd0;
    end else begin
        if (~reconfig_en[loop_counter]) begin
            if (loop_counter == NUM_DR_HOST-1) begin
                loop_counter <= 6'd0;
            end else begin
                loop_counter <= loop_counter + 1'b1;
            end
        end
    end
end

generate if (PIPELINE_ENA == 0)
begin : gen_no_pipeline
    // Direct assign signals from reconfig Host to DR
    for (genvar i=0; i<NUM_DR_HOST; i=i+1) begin : gen_out_to_host
        assign rcfg_mgmt_readdata_ch_array[i]   = (i==loop_counter & reconfig_en[i]) ? dr_avmm_readdata : {AVMM_DATA_WIDTH{1'b0}};
        assign rcfg_mgmt_readdata_valid_ch[i]   = (i==loop_counter & reconfig_en[i]) ? dr_avmm_readdata_valid : 1'b0;
        assign rcfg_mgmt_waitrequest_ch[i]      = (i==loop_counter & reconfig_en[i]) ? dr_avmm_waitrequest : 1'b1;
        assign rcfg_mgmt_new_cfg_applied_ch[i]  = (i==loop_counter & reconfig_en[i]) ? dr_new_cfg_applied : 1'b0;
    end

    assign dr_avmm_write            = reconfig_en[loop_counter] ? rcfg_mgmt_write_ch[loop_counter] : 1'b0;
    assign dr_avmm_read             = reconfig_en[loop_counter] ? rcfg_mgmt_read_ch[loop_counter] : 1'b0;
    assign dr_avmm_address          = reconfig_en[loop_counter] ? rcfg_mgmt_address_ch_array[loop_counter] : {AVMM_ADDRESS_WIDTH{1'b0}};
    assign dr_avmm_writedata        = reconfig_en[loop_counter] ? rcfg_mgmt_writedata_ch_array[loop_counter] : {AVMM_DATA_WIDTH{1'b0}};
    assign dr_new_cfg_applied_ack   = reconfig_en[loop_counter] ? rcfg_mgmt_new_cfg_applied_ack_ch[loop_counter] : 1'b0;

end else
begin : gen_pipeline
    // State machine to sequence the AVMM operations
    localparam IDLE                 = 3'b000;
    localparam READ                 = 3'b001;
    localparam READ_PASS_WAITRQ     = 3'b010;
    localparam READ_DONE            = 3'b011;
    localparam WRITE                = 3'b100;
    localparam WRITE_PASS_WAITRQ    = 3'b101;
    localparam WRITE_DONE           = 3'b110;

    logic [2:0] state;
    logic [2:0] next_state;

    always @(posedge clk or posedge reset)
    begin
        if (reset) begin
            state <= IDLE;
        end else begin
            state <= next_state;
        end
    end 

    always @ (*) begin
      case(state)
        IDLE: begin
            if (reconfig_en[loop_counter]) begin
                if (rcfg_mgmt_write_ch[loop_counter] || rcfg_mgmt_read_ch[loop_counter]) begin
                    next_state = rcfg_mgmt_write_ch[loop_counter] ? WRITE : READ;
                end else begin
                    next_state = IDLE;
                end
            end else begin
                next_state = IDLE;
            end
        end

        READ: begin
            if (dr_avmm_waitrequest) begin
                next_state = READ;
            end else begin
                next_state = READ_PASS_WAITRQ;
            end
        end

        READ_PASS_WAITRQ: begin
            next_state = READ_DONE;
        end

        READ_DONE: begin
            next_state = IDLE;
        end

        WRITE: begin
            if (dr_avmm_waitrequest) begin
                next_state = WRITE;
            end else begin
                next_state = WRITE_PASS_WAITRQ;
            end
        end

        WRITE_PASS_WAITRQ: begin
            next_state = WRITE_DONE;
        end

        WRITE_DONE: begin
            next_state = IDLE;
        end

        default : next_state = IDLE;
      endcase
    end

    // Pass signals from reconfig host to DR upon reaching certain state
    always @(posedge clk or posedge reset)
    begin
        if (reset) begin
            dr_avmm_read            <= 1'b0; 
            dr_avmm_write           <= 1'b0; 
            dr_avmm_address         <= {AVMM_ADDRESS_WIDTH{1'b0}};
            dr_avmm_writedata       <= {AVMM_DATA_WIDTH{1'b0}};
            dr_new_cfg_applied_ack  <= 1'b0;
        end else begin
            dr_new_cfg_applied_ack  <= rcfg_mgmt_new_cfg_applied_ack_ch[loop_counter];

            if (next_state == READ || next_state == WRITE) begin
                dr_avmm_address <= rcfg_mgmt_address_ch_array[loop_counter];
            end

            if(next_state == READ) begin
                dr_avmm_read  <= 1'b1; 
            end else begin
                dr_avmm_read  <= 1'b0; 
            end

            if(next_state == WRITE) begin
                dr_avmm_write  <= 1'b1; 
                dr_avmm_writedata  <= rcfg_mgmt_writedata_ch_array[loop_counter];
            end else begin 
                dr_avmm_write  <= 1'b0;
                dr_avmm_writedata  <= {AVMM_DATA_WIDTH{1'b0}};
            end
        end
    end

    // Pass signals from DR back to reconfig host upon reaching certain state
    for (genvar i=0; i<NUM_DR_HOST; i=i+1) begin : gen_out_to_host
        always @ (posedge clk or posedge reset)
        begin
            if (reset) begin
                rcfg_mgmt_new_cfg_applied_ch[i]  <= 1'b0;
                rcfg_mgmt_waitrequest_ch[i]  <= 1'b1;
                rcfg_mgmt_readdata_ch_array[i]  <= {AVMM_DATA_WIDTH{1'b0}};
                rcfg_mgmt_readdata_valid_ch[i]  <= 1'b0;
            end else begin
                rcfg_mgmt_new_cfg_applied_ch[i]  <= (i==loop_counter & reconfig_en[i]) ? dr_new_cfg_applied : 1'b0;

                if (next_state == READ_PASS_WAITRQ || next_state == WRITE_PASS_WAITRQ) begin
                    rcfg_mgmt_waitrequest_ch[i] <= (i==loop_counter & reconfig_en[i]) ? 1'b0 : 1'b1;
                end else begin
                    rcfg_mgmt_waitrequest_ch[i] <= 1'b1;
                end

                if (next_state == READ_DONE) begin
                    rcfg_mgmt_readdata_ch_array[i] <= (i==loop_counter & reconfig_en[i]) ? dr_avmm_readdata : {AVMM_DATA_WIDTH{1'b0}};
                    rcfg_mgmt_readdata_valid_ch[i] <= (i==loop_counter & reconfig_en[i]) ? 1'b1 : 1'b0;
                end else begin
                    rcfg_mgmt_readdata_valid_ch[i] <= 1'b0;
                end
            end
        end
    end
end
endgenerate

endmodule
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "+tucRC03D4QA2LL7L81sstS4hg9zUBlHnO21m59cZvuPZD5OTi6CHkbL9XLRFSGImSSDqiNZIB7Ht6hIeV1/6lX6sHaT74xmEWss1C1ppcwh4Qvfpkadj2E8tm22yhqF2txcauGsvmeSiM1YdvXMlp7vOvMbm+m/sfn+kuXHJg/4I9gLmnbC64YUyjZegghr7qJaewCdxRdkU3jqkeDVnr3EJju84C/mv9FWu6nOvS6B776V1DZHR7H9SRnqMt92Jost6cxIrhCvGZwcxgSkAiPJ1Wov7awHzvjRrJw7gWYVFTb1NyzzqUMXw/OzKgeWYgRFYAEgKzNtD0I5uzr3AOWDi6pjfUBekyNw57lby0odNUrGxeab7diXc23KtE96Zi8PVgzIZCyLaWEsEHhby4TUuJpdXwkRoCz2hcQepkFvOlqPFsmKS7MU0+z9AgxWKRdKb/dwLx1nGEFChNqDs2kKoxHVfrJrtlh9yaTzxPlgGDwKtvlpBEiqZh8jFjFLovycyfxllrJkci+XAaaG707e9bPoIBmXpKYVXdzbg4g69DfzLy0NW56fAo+raW0wZjNGfaaXytR/6FydyAKuaMf+iahoWZ7vJlRZfrh7k2QpAbzvId69OIAWCl8XrT76t9cw/7rzWC4xT/vNvGDT4tsruwxvgnrbqvpSPSQEa8AzXUl4J70ZgKh2L3QrVbjkZGsknSj2SUja6u9KLKcg5CALFa1/qvaDDdLwJNpatARzDb3pYriHpOFdhaOoNtjvX+aFpsoZX+tbxhGnNYKfM4DLoyVFSIleuoRXgt5YWsK2dOxzD2XiVA80N7cAJsbAJL6+4ZC+z5STGBfa0C7sphf3oChUSWaVA7tLpQvFTizlDFcya2mUih+tDRVonlvqb5ECR8j4Cgtp4a04d2nL8kNi0UcBbDxlcPrkpTI7s31SiZFGkWpBCOchvt8pcSnBdaCnKMU0f6NG3+BWFjW25kN2nZojKufGikM0mwgWSDWMHKJxOCMDlrKR+qSDIgRA"
`endif