// (C) 2001-2023 Intel Corporation. All rights reserved.
// Your use of Intel Corporation's design tools, logic functions and other 
// software and tools, and its AMPP partner logic functions, and any output 
// files from any of the foregoing (including device programming or simulation 
// files), and any associated documentation or information are expressly subject 
// to the terms and conditions of the Intel Program License Subscription 
// Agreement, Intel FPGA IP License Agreement, or other applicable 
// license agreement, including, without limitation, that your use is for the 
// sole purpose of programming logic devices manufactured by Intel and sold by 
// Intel or its authorized distributors.  Please refer to the applicable 
// agreement for further details.


//--------------------------------------------------------------------------------------------------
//
// Video pattern generator.
//
//--------------------------------------------------------------------------------------------------

module sdi_ii_ed_vid_pattgen #(
    parameter       OUTW_MULTP          = 1,
    parameter       SD_BIT_WIDTH        = 10,
    parameter       TEST_GEN_ANC        = 1'b0,
    parameter       TEST_GEN_VPID       = 1'b0,
    parameter [1:0] TEST_VPID_PKT_COUNT = 2'd1, // Internal test param
    parameter       TEST_ERR_VPID       = 1'b0, // Internal test param
    parameter       TEST_VPID_OVERWRITE = 1'b1  // Internal test param
) (
    input wire          clk,
    input wire          rst,
    input wire          bar_100_75n,
    input wire          enable,
    input wire          patho,
    input wire          blank,
    input wire          no_color,
    input wire          sgmt_frame, //To differentiate between 1080i50, 1080i60 and 1080sF25, 1080sF30 in tx_format == 4, 5.
    input wire  [2:0]   tx_std,
    input wire  [3:0]   tx_format,
    input wire          dl_mapping,
    input wire          ntsc_paln,

    output reg  [OUTW_MULTP*20-1:0] dout,
    output reg                      dout_valid,
    output reg                      trs,
    output reg  [OUTW_MULTP*11-1:0] ln,
    output reg  [19:0]              dout_b,
    output reg                      dout_valid_b,
    output reg                      trs_b,
    output reg  [OUTW_MULTP*11-1:0] ln_b,
    output reg  [OUTW_MULTP*8-1:0]  vpid_byte1,
    output reg  [OUTW_MULTP*8-1:0]  vpid_byte2,
    output reg  [OUTW_MULTP*8-1:0]  vpid_byte3,
    output reg  [OUTW_MULTP*8-1:0]  vpid_byte4,
    output reg  [OUTW_MULTP*8-1:0]  vpid_byte1_b,
    output reg  [OUTW_MULTP*8-1:0]  vpid_byte2_b,
    output reg  [OUTW_MULTP*8-1:0]  vpid_byte3_b,
    output reg  [OUTW_MULTP*8-1:0]  vpid_byte4_b,
    output wire [OUTW_MULTP*11-1:0] line_f0,
    output wire [OUTW_MULTP*11-1:0] line_f1
  );

localparam       EN_SD_20BITS = (SD_BIT_WIDTH == 20);
localparam [4:0] WORD_DIVIDER = `ifdef SDI_SIM 5'd20; `else 5'd1; `endif
localparam [2:0] HANC_DIVIDER = `ifdef SDI_SIM 3'd5; `else 3'd1; `endif

wire [7:0]  int_sd_hanc_y_word;
wire [11:0] int_hd_hanc_word;
wire [10:0] int_lines_per_frame;
wire [12:0] int_words_per_active_line;
wire [12:0] int_words_per_total_line;
wire [10:0] int_f_rise_line;
wire [10:0] int_f_fall_line;
wire [10:0] int_v_fall_line_1;
wire [10:0] int_v_rise_line_1;
wire [10:0] int_v_fall_line_2;
wire [10:0] int_v_rise_line_2;
wire [10:0] int_patho_change_line_1;
wire [10:0] int_patho_change_line_2;
wire [7:0]  int_vpid_byte1;
wire [7:0]  int_vpid_byte2_a;
wire [7:0]  int_vpid_byte2;
wire [7:0]  int_vpid_byte3;
wire [7:0]  int_vpid_byte4;
wire [10:0] int_line_f0;
wire [10:0] int_line_f1;

wire        hd_sdn;
wire [1:0]  tx_std_int;

assign tx_std_int = {|tx_std[2:1], tx_std[0]};
assign hd_sdn = tx_std_int[1] ? 1'b1 : (tx_std_int[0] ? 1'b1 : 1'b0);

////////////////////
wire        int_sd_en;
reg  [19:0] t_dout_int_i;
reg  [19:0] dout_int_i;
wire [19:0] dout_i;
wire        dout_valid_i;
wire        trs_20;
wire [19:0] dout_sd;

generate if ( EN_SD_20BITS ) 
begin: sd_10bit_ce_gen

  reg [3:0] sdcnt;
  always @ (posedge clk or posedge rst)
  begin
    if (rst) begin
      sdcnt <= 4'd0;
    end else begin
      if (tx_std_int == 2'b00) begin
        if (sdcnt != 4'd11) begin
          sdcnt <= sdcnt + 4'd1;
        end else begin
          sdcnt <= 4'd1;
        end  
      end else begin
        sdcnt <= 4'd0;
      end
      
     end
  end
  assign int_sd_en = (sdcnt == 4'd1 | sdcnt == 4'd6);
  
  always @ (posedge clk or posedge rst)
  begin
    if (rst) begin
      t_dout_int_i <= 20'd0;
      dout_int_i <= 20'd0;
    end else begin
      if (dout_valid_i) begin
        t_dout_int_i <= {dout_i[9:0], t_dout_int_i[19:10]};
      end

      if (enable) begin
        dout_int_i <= t_dout_int_i;
      end
    end
  end
  
  assign trs_20 = (dout_int_i == 20'h003ff);
  assign dout_sd = dout_int_i;
  
end else begin

  assign int_sd_en = 1'b0;
  assign trs_20 = 1'b0;
  assign dout_sd = dout_i;

end
endgenerate

//////////////

// ---------------------
// Clock enable for 3GB
// ---------------------
reg ce_3gb;
always @ (posedge clk or posedge rst)
begin
  if (rst) ce_3gb <= 1'b0;
   else     ce_3gb <= ~ce_3gb;
end

wire ce_i;
assign ce_i = (tx_std_int==2'b10) ? ce_3gb : ((tx_std_int != 2'b00) ? enable : (EN_SD_20BITS ? int_sd_en : enable));

//--------------------------------------------------------------------------------------------------
// Colorbar generator
//--------------------------------------------------------------------------------------------------
wire req;                               // Data request from frame generator to pattern generator
wire [12:0] wn;                         // Word number to accompany request
wire [9:0] y_gen;
wire [9:0] c_gen;
sdi_ii_colorbar_gen u_colorbar (
  .clk                      (clk),
  .rst                      (rst),
  .hd_sdn                   (hd_sdn),
  .bar_100_75n              (bar_100_75n),
  .req                      (req),
  .wn                       (wn),
  .words_per_active_line    (int_words_per_active_line),
  .yout                     (y_gen),
  .cout                     (c_gen)
  );

//--------------------------------------------------------------------------------------------------
// Pathological pattern generator
//--------------------------------------------------------------------------------------------------
wire [9:0] y_patho;
wire [9:0] c_patho;
wire [10:0] ln_i;
sdi_ii_patho_gen u_patho (
  .hd_sdn                   (hd_sdn),
  .clk                      (clk),
  .rst                      (rst),
  .req                      (req),
  .ln                       (ln_i),
  .wn                       (wn),
  .yout                     (y_patho),
  .cout                     (c_patho),
  .field1_start_ln          (tx_std_int==2'b10 & dl_mapping ? 11'd21 : int_v_fall_line_1),
  .field1_pattern_change_ln (tx_std_int==2'b10 & dl_mapping ? 11'd290 : int_patho_change_line_1),
  .field2_start_ln          (tx_std_int==2'b10 & dl_mapping ? 11'd584 : int_v_fall_line_2),
  .field2_pattern_change_ln (tx_std_int==2'b10 & dl_mapping ? 11'd853 : int_patho_change_line_2),
  .pal_nstcn                (tx_format[0])
  );

//--------------------------------------------------------------------------------------------------
// Select data for output pattern - blank, pathological or colorbar
//--------------------------------------------------------------------------------------------------

wire [9:0] y_make = patho ? y_patho : blank ? 10'h040 : y_gen;
wire [9:0] c_make = patho ? c_patho : (no_color | blank) ? 10'h200 : c_gen;


//--------------------------------------------------------------------------------------------------
// Create output frame - 1080i used for this demo
//--------------------------------------------------------------------------------------------------
wire [10:0] lnb_i;
wire [10:0] dl_ln;
wire        trs_i;
//wire        anc_i;

sdi_ii_makeframe u_makeframe (
  .hd_sdn                   (hd_sdn),

  .clk                      (clk),
  .rst                      (rst),
  .enable                   (ce_i),
  .din_req                  (req),
  .ln                       (ln_i),
  .lnB                      (lnb_i),
  .dl_ln                    (dl_ln),  
  .tx_std                   (tx_std_int),
  .dl_mapping               (dl_mapping),
  //.field_line             (),
  .word_count               (wn),
  .din_y                    (y_make),
  .din_c                    (c_make),
  .dout                     (dout_i),
  .dout_valid               (dout_valid_i),
  .trs                      (trs_i),
  .anc                      (),
  
  .sd_hanc_y_word           (int_sd_hanc_y_word),
  .hd_hanc_word             (int_hd_hanc_word),
  .lines_per_frame          (int_lines_per_frame),
  .words_per_active_line    (int_words_per_active_line),
  .words_per_total_line     (int_words_per_total_line),
  .f_rise_line              (int_f_rise_line),
  .f_fall_line              (int_f_fall_line),
  .v_fall_line_1            (int_v_fall_line_1),
  .v_rise_line_1            (int_v_rise_line_1),
  .v_fall_line_2            (int_v_fall_line_2),
  .v_rise_line_2            (int_v_rise_line_2),
  .vpid_byte1               (int_vpid_byte1),
  .vpid_byte2               (int_vpid_byte2),
  .vpid_byte3               (int_vpid_byte3),
  .vpid_byte4               (int_vpid_byte4),
  .line_f0                  (int_line_f0),
  .line_f1                  (int_line_f1),
  .vpid_byte1_b              (vpid_byte1_b[7:0]),
  .vpid_byte2_b              (vpid_byte2_b[7:0]),
  .vpid_byte3_b              (vpid_byte3_b[7:0]),
  .vpid_byte4_b              (vpid_byte4_b[7:0])
  );
  defparam u_makeframe.TEST_GEN_ANC        = TEST_GEN_ANC;
  defparam u_makeframe.TEST_GEN_VPID       = TEST_GEN_VPID;
  defparam u_makeframe.TEST_VPID_PKT_COUNT = TEST_VPID_PKT_COUNT;
  defparam u_makeframe.TEST_ERR_VPID       = TEST_ERR_VPID;
  defparam u_makeframe.TEST_VPID_OVERWRITE = TEST_VPID_OVERWRITE;

// ------------------------------------------------------------------------------
// Turn on "Allow any ROM size for recognition' to infer ROM for the logic below
// ------------------------------------------------------------------------------
reg  [186:0] ROM [0:15];
wire [3:0]   ADDRA = 4'b0000;
wire         WEA   = 1'b0;
wire [186:0] DATAA = 187'd0;

//To resolve DA Warning - RAM Control Signals Driven by Flip-Flops with Asynchronous Clears.
//assign RADDR = tx_format;
                                                                                                                                                                                   // FORMAT                   SUPPORTED INTERFACE 
initial begin                                                                                                                         //HD,   3GA,  3GB,  byte2  line_f0 line_f1   // --------------------------------------------------------
   ROM[0]   <= {8'd134, 12'd   0, 11'd525,   13'd720,  13'd858,  11'd266, 11'd4, 11'd20, 11'd264,  11'd283, 11'd1,    11'd140, 11'd400, 4'h1, 4'h0, 4'h0, 8'h06, 11'd13, 11'd276}; // NTSC                     259M SD
   ROM[1]   <= {8'd140, 12'd   0, 11'd625,   13'd720,  13'd864,  11'd313, 11'd1, 11'd23, 11'd311,  11'd336, 11'd624,  11'd160, 11'd470, 4'h1, 4'h0, 4'h0, 8'h05, 11'd9,  11'd322}; // PAL                      259M SD
   ROM[2]   <= {8'd  0, 12'd 268, 11'd1125,  13'd1920, 13'd2200, 11'd564, 11'd1, 11'd41, 11'd1121, 11'd603, 11'd558,  11'd298, 11'd861, 4'h5, 4'h9, 4'hC, 8'h07, 11'd10, 11'd572}; // 1035i                    260M HD,3GB-2x1080-line                                    
   ROM[3]   <= {8'd  0, 12'd 444, 11'd1250,  13'd1920, 13'd2376, 11'd626, 11'd1, 11'd81, 11'd621,  11'd706, 11'd1246, 11'd350, 11'd975, 4'h5, 4'h9, 4'hC, 8'h05, 11'd10, 11'd572}; // 1080i50                  295M HD,3GB-2x1080-line   WFM700/7120 can't recognize this
   ROM[4]   <= {8'd  0, 12'd 268, 11'd1125,  13'd1920, 13'd2200, 11'd564, 11'd1, 11'd21, 11'd561,  11'd584, 11'd1124, 11'd290, 11'd853, 4'h5, 4'h9, 4'hC, 8'h07, 11'd10, 11'd572}; // 1080i60/59.94            274M HD,3GB-2x1080-line
   ROM[5]   <= {8'd  0, 12'd 708, 11'd1125,  13'd1920, 13'd2640, 11'd564, 11'd1, 11'd21, 11'd561,  11'd584, 11'd1124, 11'd290, 11'd853, 4'h5, 4'h9, 4'hC, 8'h05, 11'd10, 11'd572}; // 1080i50                  274M HD,3GB-2x1080-line
   ROM[6]   <= {8'd  0, 12'd 818, 11'd1125,  13'd1920, 13'd2750, 11'd0,   11'd0, 11'd42, 11'd1122, 11'd0,   11'd0,    11'd581, 11'd0,   4'h5, 4'h9, 4'hC, 8'hC3, 11'd10, 11'd0};   // 1080p24/23.98            274M HD,3GB-2x1080-line
   ROM[7]   <= {8'd  0, 12'd 358, 11'd750,   13'd1280, 13'd1650, 11'd0,   11'd0, 11'd26, 11'd746,  11'd0,   11'd0,    11'd385, 11'd0,   4'h4, 4'h8, 4'hB, 8'h4B, 11'd10, 11'd0};   // 720p60/59.94             296M HD,3GB-2x720-line
   ROM[8]   <= {8'd  0, 12'd 688, 11'd750,   13'd1280, 13'd1980, 11'd0,   11'd0, 11'd26, 11'd746,  11'd0,   11'd0,    11'd385, 11'd0,   4'h4, 4'h8, 4'hB, 8'h49, 11'd10, 11'd0};   // 720p50                   296M HD,3GB-2x720-line
   ROM[9]   <= {8'd  0, 12'd2008, 11'd750,   13'd1280, 13'd3300, 11'd0,   11'd0, 11'd26, 11'd746,  11'd0,   11'd0,    11'd385, 11'd0,   4'h4, 4'h8, 4'hB, 8'h47, 11'd10, 11'd0};   // 720p30/29.97             296M HD,3GB-2x720-line
   ROM[10]  <= {8'd  0, 12'd2668, 11'd750,   13'd1280, 13'd3960, 11'd0,   11'd0, 11'd26, 11'd746,  11'd0,   11'd0,    11'd385, 11'd0,   4'h4, 4'h8, 4'hB, 8'h45, 11'd10, 11'd0};   // 720p25                   296M HD,3GB-2x720-line
   ROM[11]  <= {8'd  0, 12'd2833, 11'd750,   13'd1280, 13'd4125, 11'd0,   11'd0, 11'd26, 11'd746,  11'd0,   11'd0,    11'd385, 11'd0,   4'h4, 4'h8, 4'hB, 8'h43, 11'd10, 11'd0};   // 720p24/23.98             296M HD,3GB-2x720-line
   ROM[12]  <= {8'd  0, 12'd 268, 11'd1125,  13'd1920, 13'd2200, 11'd0,   11'd0, 11'd42, 11'd1122, 11'd0,   11'd0,    11'd581, 11'd0,   4'h5, 4'h9, 4'hC, 8'hC7, 11'd10, 11'd0};   // 1080p30/29.97,60/59.94   274M HD(30/29.97),,HD-DL(60/59.94),3GA(60/59.94),3GB-DL(60/59.94)
   ROM[13]  <= {8'd  0, 12'd 708, 11'd1125,  13'd1920, 13'd2640, 11'd0,   11'd0, 11'd42, 11'd1122, 11'd0,   11'd0,    11'd581, 11'd0,   4'h5, 4'h9, 4'hC, 8'hC5, 11'd10, 11'd0};   // 1080p25,50               274M HD(25),HD-DL(50),3GA(50),3GB-DL(50)
   ROM[14]  <= {8'd  0, 12'd 818, 11'd1125,  13'd1920, 13'd2750, 11'd564, 11'd1, 11'd21,  11'd561, 11'd584, 11'd1124, 11'd290, 11'd853, 4'h5, 4'h9, 4'hC, 8'h43, 11'd10, 11'd572}; // 1080sF24/23.98           274M HD,3GB-2x1080-line
   ROM[15]  <= {8'd0, 12'd0, 11'd0, 13'd0, 13'd0, 11'd0, 11'd0, 11'd0, 11'd0, 11'd0, 11'd0, 11'd0, 11'd0, 4'h0, 4'h0, 4'h0, 8'h00, 11'd10, 11'd0};    // Reserved
end

always @ (posedge clk)
begin
  if (WEA) begin
    ROM[ADDRA] <= DATAA;
  end
end

reg   [3:0] RADDR;
reg [186:0] RDATA;
always @ (posedge clk)
begin
  RADDR <= tx_format;
  RDATA <= ROM[RADDR];
end

wire [3:0] rdata1;
wire [3:0] rdata2;
assign rdata1 = tx_std_int[1] ? (tx_std_int[0] ? RDATA[37:34] : (dl_mapping ? 4'hA : RDATA[33:30])) : 
                                (tx_std_int[0] ? (dl_mapping ? 4'h7 : RDATA[41:38]) : RDATA[41:38]);

assign rdata2 = (tx_format == 4'd12 | tx_format == 4'd13) ? {~dl_mapping, RDATA[28:26]} : 
                (tx_format == 4'd4  | tx_format == 4'd5)  ? {1'b0, sgmt_frame, RDATA[27:26]} : RDATA[29:26];

assign int_sd_hanc_y_word        = RDATA[186:179]/HANC_DIVIDER; // words_per_total_line - words_per_active_line - 4
assign int_hd_hanc_word          = RDATA[178:167]/HANC_DIVIDER; // words_per_total_line - words_per_active_line - 8 - 4
assign int_lines_per_frame       = RDATA[166:156];
assign int_words_per_active_line = RDATA[155:143]/WORD_DIVIDER;
assign int_words_per_total_line  = RDATA[142:130]/WORD_DIVIDER;
assign int_f_rise_line           = RDATA[129:119];
assign int_f_fall_line           = RDATA[118:108];
assign int_v_fall_line_1         = RDATA[107:97];
assign int_v_rise_line_1         = RDATA[96:86];
assign int_v_fall_line_2         = RDATA[85:75];
assign int_v_rise_line_2         = RDATA[74:64];
assign int_patho_change_line_1   = RDATA[63:53];
assign int_patho_change_line_2   = RDATA[52:42];
assign int_vpid_byte1            = tx_std == 3'b111 ? 8'hCE :
                                   tx_std == 3'b100 ? 8'hC0 :
                                                     {4'b1000, rdata1};

                                    // Picture rate is doubled for 3G and HD Dual Link interfaces
assign int_vpid_byte2_a          = (tx_std_int==2'b11 | dl_mapping) ? {rdata2, (tx_format==4'd12 ? 4'hB : (tx_format==4'd13 ? 4'h9 : RDATA[25:22]))} :
                                                                      {rdata2, RDATA[25:22]};
assign int_vpid_byte2            = (ntsc_paln & int_vpid_byte2_a[3:0] == 4'h3) ? {int_vpid_byte2_a[7:4], 4'h2} :
                                   (ntsc_paln & int_vpid_byte2_a[3:0] == 4'h7) ? {int_vpid_byte2_a[7:4], 4'h6} :
                                   (ntsc_paln & int_vpid_byte2_a[3:0] == 4'hB) ? {int_vpid_byte2_a[7:4], 4'hA} :
                                                                                 int_vpid_byte2_a;
assign int_vpid_byte3            = 8'b00000000; // Sampling structure = YCrCb 422
assign int_vpid_byte4            = 8'b00000001; // Bit depth = 10
assign int_line_f0               = RDATA[21:11];
assign int_line_f1               = RDATA[10:0];

// ----------------------
// Interleave 3GB output
// ----------------------
reg [19:0] dout_3gb, dout_3gb_dly;
reg [1:0]  trs_dly;
reg        ce_i_dly;
always @ (posedge clk or posedge rst) // frequency of 148.5MHz 
begin
  if (rst) begin
    trs_dly      <= 2'b00;
    dout_3gb     <= 20'd0;
    dout_3gb_dly <= 20'd0;
    ce_i_dly     <= 1'b0;
  end else begin
    trs_dly[0]   <= trs_i;
    trs_dly[1]   <= trs_dly[0];
    dout_3gb_dly <= dout_3gb;
    ce_i_dly     <= ce_i;

    if (ce_i_dly) begin // frequency of 74.25MHz
      dout_3gb[19:10] <= dout_i[19:10];
      dout_3gb[9:0]   <= dout_i[19:10];
    end else begin
      dout_3gb[19:10] <= dout_i[9:0];
      dout_3gb[9:0]   <= dout_i[9:0];
    end
  end
end

always @ (posedge clk or posedge rst)
begin
    if (rst) begin
        dout <= {OUTW_MULTP*20{1'b0}};
        dout_b <= 20'd0;
        dout_valid <= 1'b0;
        dout_valid_b <= 1'b0;
        trs <= 1'b0;
        trs_b <= 1'b0;
        ln <= {OUTW_MULTP*11{1'b0}};
        ln_b <= {OUTW_MULTP*11{1'b0}};
        vpid_byte1 <= {OUTW_MULTP*8{1'b0}};
        vpid_byte2 <= {OUTW_MULTP*8{1'b0}};
        vpid_byte3 <= {OUTW_MULTP*8{1'b0}};
        vpid_byte4 <= {OUTW_MULTP*8{1'b0}};
        vpid_byte1_b <= {OUTW_MULTP*8{1'b0}};
        vpid_byte2_b <= {OUTW_MULTP*8{1'b0}};
        vpid_byte3_b <= {OUTW_MULTP*8{1'b0}};
        vpid_byte4_b <= {OUTW_MULTP*8{1'b0}};
    end else begin
        dout_b <= dout_i;
        dout_valid_b <= dout_valid_i;
        trs_b <= trs_i;
        vpid_byte1 <= {OUTW_MULTP{int_vpid_byte1}};
        vpid_byte2 <= {OUTW_MULTP{int_vpid_byte2}};
        vpid_byte3 <= {OUTW_MULTP{int_vpid_byte3}};
        vpid_byte4 <= {OUTW_MULTP{int_vpid_byte4}};
        vpid_byte1_b <= {OUTW_MULTP{int_vpid_byte1}};
        vpid_byte2_b <= {OUTW_MULTP{int_vpid_byte2}};
        vpid_byte3_b <= {OUTW_MULTP{int_vpid_byte3}};

        if (tx_std_int == 2'b10) begin          //3GB
            dout <= {OUTW_MULTP{dout_3gb_dly}};
            dout_valid <= enable;
            trs <= trs_dly[1];
            ln <= {OUTW_MULTP{ln_i}};
            ln_b <= {OUTW_MULTP{lnb_i}};
            if (dl_mapping) begin
                vpid_byte4_b <= {OUTW_MULTP{int_vpid_byte4[7], 1'b1, int_vpid_byte4[5:0]}}; // Channel assignment: 1-link A, 2-linkB
            end else begin
                vpid_byte4_b <= {OUTW_MULTP{int_vpid_byte4[7], 1'b0, int_vpid_byte4[5:0]}};
            end
        end else if (tx_std_int == 2'b00) begin     //SD
            dout <= {OUTW_MULTP{dout_sd}};
            ln <= {OUTW_MULTP{ln_i}};
            ln_b <= {OUTW_MULTP{lnb_i}};
            vpid_byte4_b <= {OUTW_MULTP{int_vpid_byte4}};
            if (EN_SD_20BITS) begin
                dout_valid <= enable;
                trs <= trs_20;
            end else begin
                dout_valid <= dout_valid_i;
                trs <= trs_i;
            end
        end else if (tx_std_int == 2'b01) begin     //HD
            dout <= {OUTW_MULTP{dout_i}};
            dout_valid <= dout_valid_i;
            trs <= trs_i;
            if (dl_mapping) begin
                vpid_byte4_b <= {OUTW_MULTP{int_vpid_byte4[7], 1'b1, int_vpid_byte4[5:0]}};
                ln <= {OUTW_MULTP{dl_ln}};
                ln_b <= {OUTW_MULTP{dl_ln}};
            end else begin
                vpid_byte4_b <= {OUTW_MULTP{int_vpid_byte4[7], 1'b0, int_vpid_byte4[5:0]}};
                ln <= {OUTW_MULTP{ln_i}};
                ln_b <= {OUTW_MULTP{lnb_i}};
            end
        end else begin                          //3GA
            dout <= {OUTW_MULTP{dout_i}};
            dout_valid <= enable;
            trs <= trs_i;
            ln <= {OUTW_MULTP{ln_i}};
            ln_b <= {OUTW_MULTP{lnb_i}};
            vpid_byte4_b <= {OUTW_MULTP{int_vpid_byte4}};
        end
    end
end

assign line_f0    = {OUTW_MULTP{int_line_f0}};
assign line_f1    = {OUTW_MULTP{int_line_f1}};

endmodule
`ifdef QUESTA_INTEL_OEM
`pragma questa_oem_00 "MBeNMpYGPtPl1sE+DQYudSgTsK7EAWNJwUGKRF0u7XRPhowIlls8RJfstyWmsvlOEmdSGdceoRHeaHEQLNzQQIhfBlevOTODcoCezHsI0TKITCVzhv7PoO/WuhggyXiWx6PZ9QL2w6wOssoF+/IIDkKjb9O37NLgtiIz/Y8rj7aO4WI3Z30kcN5p8CvKXADJL6/obbPPFES6e1ypucbAQpHseQQQWhKLHWIfs3d6tL4RuK+H0DJFlIEEaXzPE89bV9MCnRmOTFOkpF6gE8NnEDI6K52uERg5+OFjzFDKwkrNKuL3IPriL5dFImremLUbiD2NtxG+ykrdsMM56KzasuA6C2KHZk0RCUcafbUf9p8k0aD+yClWXbukqQu5qKMKDQYg9P/iMDS4agFftulB+vU4+mYjim2vyGRvs3reL1oBJU0flI1Ymf9LoZoynBKCkLZFMgYXcwgEUSAt62YjARlHnjWKANPRPpm3oC2gr51wLnYzVkdCcYCAvfOEA83zlIUYgRI1eEwiv6uJbfh3QWw6Q72fGJ76mCkP5eiHu+wefEvTq2gyyPdAABtSqK7a0geENBFXcm99nwB3Rnui7i5TUqyYGjVTiRBoMMoqPS7UPQtKCi+lMIGZQJeRB3b6kEgyoy2HpunXV0xvmzCAZOCJfyKV/gXumEW/u8Qze4br/4PUJ3rGdUS7iGWcpuDdPpJxoP8ps5I4cPl9Zo4S3r41NPxWLxp3L1G0xjf78MEviMqbS05akUX7RcLS3muyhGNRZ7l3fGprnFJNubWgqf/yjBz2qUGKaTXU+xdZnLq5YxjgJPZ2seehOCazTMEd/5//knGzE3oydvRskzmPfMRbtJIv787UrsWCBrPMJNCezQwEKNi9ldIu7BvieAqUUQoO/xDH1gSmkp61I6zYZizh232SvZOg2tf3RP9QPH1SrFOyKxc4MfR1QTZSuivjsRRBeH38W01TBAlIVbuZH65eZeew0HQjG18bMkgowrwuwreNEO00kKB84A1vxB87"
`endif